/*  Copyright (C) CZ.NIC, z.s.p.o. and contributors
 *  SPDX-License-Identifier: GPL-2.0-or-later
 *  For more information, see <https://www.knot-dns.cz/>
 */

#pragma once

#include "libdnssec/binary.h"

typedef struct key_parameters {
	// DNSSEC fields
	uint8_t *name;
	uint16_t flags;
	uint8_t protocol;
	uint8_t algorithm;
	dnssec_binary_t public_key;

	// DNSSEC wire format
	dnssec_binary_t rdata;

	// Hashes
	const char *key_id;
	uint16_t keytag;
	dnssec_binary_t ds_sha1;
	dnssec_binary_t ds_sha256;
	dnssec_binary_t ds_sha384;

	// Key information
	unsigned bit_size;

	// Private key in PEM
	dnssec_binary_t pem;
} key_parameters_t;

/*

RSASHA256 (1024b)

rsa1024-256.	DNSKEY	  256  3  8  AwEAAaqwL+O6GcCPkRZjoObfIJHcHPwQQY9mnAg6kROea2gsyRJOAwBNQPCfXoPtmrU0BiZ0aGBVTVPAvZh+HJRu9NEfTNDPK2HSyHdSucjY1qs6WAub6oWHJuLBxMesftpnUwoLnVZyN+pOblUZUMsvxP3PlS+mA+E6NyQX0F/fcfGL
rsa1024-256.	DS    	37335  8  1  1039E8882D516C0BA265C9BC981EE05B9DC579E6
rsa1024-256.	DS    	37335  8  2  7080B51BB07449A1858322911B53BC05C26BDCE79C5B0F283CE6A829EB766FEC
rsa1024-256.	DS    	37335  8  4  A09F6B14A3FD17EA2D49899467ACC0A877E572246ED9744E280172C5E27F0D2935F8B8CECBD91D0CB044E043A6EE76A4

-----BEGIN PRIVATE KEY-----
MIICdgIBADANBgkqhkiG9w0BAQEFAASCAmAwggJcAgEAAoGBAKqwL+O6GcCPkRZj
oObfIJHcHPwQQY9mnAg6kROea2gsyRJOAwBNQPCfXoPtmrU0BiZ0aGBVTVPAvZh+
HJRu9NEfTNDPK2HSyHdSucjY1qs6WAub6oWHJuLBxMesftpnUwoLnVZyN+pOblUZ
UMsvxP3PlS+mA+E6NyQX0F/fcfGLAgMBAAECgYA0YNKhUEd8xtGgDDyg/FGTUPd4
BkkIB1XUVJyWZ6PQGQYu3s+e9RX7gPxRAcXa2zcT797jyOlszRWhSPneNtpbHVQt
WZDVykljRcuxFSZpB07ASdKBeO1DLaG2W2HxGKcH4udUbx5RL2NacQSKR+8IsfSm
2Q5r+7oYoVQyP5D3mQJBAOGWY/2A+cb48RLhNLkdCnnbGXhk+oh6sLZibizboRWn
lyMOb112ZoPPFP5SKNrdAxEmE2314JcNHrutjMFm/ucCQQDBsxTTAzoxhoOzgV8R
65NdPXg/dxdk4UaKQk9aIhQagwa8OBjPF2IWgfMZ0uRqmw+LNOoEoy1hD0JhBryD
Vxe9AkEAywKuZVqGbdtmB9mHuvc5kEPuffxRwjS3hsq538CfDH/PcYryCagdxYy8
lcqWXa/7rJkZbyGQxh7Wg4tBWmM4DQJAL8MYv29sSgoBL6IW7zRHghZGMGANRLLH
0g/HwVHl4yOr5X1voKEDbslcSGHYMPFLQ+goTDxwVB6PH52pnjk7gQJAUSHiV/UQ
kTz3BlxZc1IAiUQv9/Ba8wtHWSVp+YqPhxt1sfdiyUMXtlA4f6WAKAGMraoRw4wI
cYr+N6Wx+wwXZw==
-----END PRIVATE KEY-----

*/
static const key_parameters_t SAMPLE_RSA1024_SHA256_KEY = {
	.name = (uint8_t *)"\x0b""rsa1024-256",
	.flags = 256,
	.protocol = 3,
	.algorithm = 8,
	.public_key = { .size = 132, .data = (uint8_t []){
		0x03, 0x01, 0x00, 0x01, 0xaa, 0xb0, 0x2f, 0xe3, 0xba, 0x19,
		0xc0, 0x8f, 0x91, 0x16, 0x63, 0xa0, 0xe6, 0xdf, 0x20, 0x91,
		0xdc, 0x1c, 0xfc, 0x10, 0x41, 0x8f, 0x66, 0x9c, 0x08, 0x3a,
		0x91, 0x13, 0x9e, 0x6b, 0x68, 0x2c, 0xc9, 0x12, 0x4e, 0x03,
		0x00, 0x4d, 0x40, 0xf0, 0x9f, 0x5e, 0x83, 0xed, 0x9a, 0xb5,
		0x34, 0x06, 0x26, 0x74, 0x68, 0x60, 0x55, 0x4d, 0x53, 0xc0,
		0xbd, 0x98, 0x7e, 0x1c, 0x94, 0x6e, 0xf4, 0xd1, 0x1f, 0x4c,
		0xd0, 0xcf, 0x2b, 0x61, 0xd2, 0xc8, 0x77, 0x52, 0xb9, 0xc8,
		0xd8, 0xd6, 0xab, 0x3a, 0x58, 0x0b, 0x9b, 0xea, 0x85, 0x87,
		0x26, 0xe2, 0xc1, 0xc4, 0xc7, 0xac, 0x7e, 0xda, 0x67, 0x53,
		0x0a, 0x0b, 0x9d, 0x56, 0x72, 0x37, 0xea, 0x4e, 0x6e, 0x55,
		0x19, 0x50, 0xcb, 0x2f, 0xc4, 0xfd, 0xcf, 0x95, 0x2f, 0xa6,
		0x03, 0xe1, 0x3a, 0x37, 0x24, 0x17, 0xd0, 0x5f, 0xdf, 0x71,
		0xf1, 0x8b,
	}},
	.rdata = { .size = 136, .data = (uint8_t []){
		01, 0x00, 0x03, 0x08,
		0x03, 0x01, 0x00, 0x01, 0xaa, 0xb0, 0x2f, 0xe3, 0xba, 0x19,
		0xc0, 0x8f, 0x91, 0x16, 0x63, 0xa0, 0xe6, 0xdf, 0x20, 0x91,
		0xdc, 0x1c, 0xfc, 0x10, 0x41, 0x8f, 0x66, 0x9c, 0x08, 0x3a,
		0x91, 0x13, 0x9e, 0x6b, 0x68, 0x2c, 0xc9, 0x12, 0x4e, 0x03,
		0x00, 0x4d, 0x40, 0xf0, 0x9f, 0x5e, 0x83, 0xed, 0x9a, 0xb5,
		0x34, 0x06, 0x26, 0x74, 0x68, 0x60, 0x55, 0x4d, 0x53, 0xc0,
		0xbd, 0x98, 0x7e, 0x1c, 0x94, 0x6e, 0xf4, 0xd1, 0x1f, 0x4c,
		0xd0, 0xcf, 0x2b, 0x61, 0xd2, 0xc8, 0x77, 0x52, 0xb9, 0xc8,
		0xd8, 0xd6, 0xab, 0x3a, 0x58, 0x0b, 0x9b, 0xea, 0x85, 0x87,
		0x26, 0xe2, 0xc1, 0xc4, 0xc7, 0xac, 0x7e, 0xda, 0x67, 0x53,
		0x0a, 0x0b, 0x9d, 0x56, 0x72, 0x37, 0xea, 0x4e, 0x6e, 0x55,
		0x19, 0x50, 0xcb, 0x2f, 0xc4, 0xfd, 0xcf, 0x95, 0x2f, 0xa6,
		0x03, 0xe1, 0x3a, 0x37, 0x24, 0x17, 0xd0, 0x5f, 0xdf, 0x71,
		0xf1, 0x8b,
	}},
	.key_id = "76f0d6c093d8328bc7f0e25bd8cde5575bad9b44",
	.keytag = 37335,
	.ds_sha1 = { .size = 24, .data = (uint8_t []){
		0x91, 0xd7, 0x08, 0x01, 0x10, 0x39, 0xe8, 0x88, 0x2d, 0x51,
		0x6c, 0x0b, 0xa2, 0x65, 0xc9, 0xbc, 0x98, 0x1e, 0xe0, 0x5b,
		0x9d, 0xc5, 0x79, 0xe6,
	}},
	.ds_sha256 = { .size = 36, .data = (uint8_t []){
		0x91, 0xd7, 0x08, 0x02, 0x70, 0x80, 0xb5, 0x1b, 0xb0, 0x74,
		0x49, 0xa1, 0x85, 0x83, 0x22, 0x91, 0x1b, 0x53, 0xbc, 0x05,
		0xc2, 0x6b, 0xdc, 0xe7, 0x9c, 0x5b, 0x0f, 0x28, 0x3c, 0xe6,
		0xa8, 0x29, 0xeb, 0x76, 0x6f, 0xec,
	}},
	.ds_sha384 = { .size = 52, .data = (uint8_t []){
		0x91, 0xd7, 0x08, 0x04, 0xa0, 0x9f, 0x6b, 0x14, 0xa3, 0xfd,
		0x17, 0xea, 0x2d, 0x49, 0x89, 0x94, 0x67, 0xac, 0xc0, 0xa8,
		0x77, 0xe5, 0x72, 0x24, 0x6e, 0xd9, 0x74, 0x4e, 0x28, 0x01,
		0x72, 0xc5, 0xe2, 0x7f, 0x0d, 0x29, 0x35, 0xf8, 0xb8, 0xce,
		0xcb, 0xd9, 0x1d, 0x0c, 0xb0, 0x44, 0xe0, 0x43, 0xa6, 0xee,
		0x76, 0xa4,
	}},
	.bit_size = 1024,
	.pem = { .size = 916, .data = (uint8_t []){
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49,
		0x49, 0x43, 0x64, 0x67, 0x49, 0x42, 0x41, 0x44, 0x41, 0x4e,
		0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77,
		0x30, 0x42, 0x41, 0x51, 0x45, 0x46, 0x41, 0x41, 0x53, 0x43,
		0x41, 0x6d, 0x41, 0x77, 0x67, 0x67, 0x4a, 0x63, 0x41, 0x67,
		0x45, 0x41, 0x41, 0x6f, 0x47, 0x42, 0x41, 0x4b, 0x71, 0x77,
		0x4c, 0x2b, 0x4f, 0x36, 0x47, 0x63, 0x43, 0x50, 0x6b, 0x52,
		0x5a, 0x6a, 0x0a, 0x6f, 0x4f, 0x62, 0x66, 0x49, 0x4a, 0x48,
		0x63, 0x48, 0x50, 0x77, 0x51, 0x51, 0x59, 0x39, 0x6d, 0x6e,
		0x41, 0x67, 0x36, 0x6b, 0x52, 0x4f, 0x65, 0x61, 0x32, 0x67,
		0x73, 0x79, 0x52, 0x4a, 0x4f, 0x41, 0x77, 0x42, 0x4e, 0x51,
		0x50, 0x43, 0x66, 0x58, 0x6f, 0x50, 0x74, 0x6d, 0x72, 0x55,
		0x30, 0x42, 0x69, 0x5a, 0x30, 0x61, 0x47, 0x42, 0x56, 0x54,
		0x56, 0x50, 0x41, 0x76, 0x5a, 0x68, 0x2b, 0x0a, 0x48, 0x4a,
		0x52, 0x75, 0x39, 0x4e, 0x45, 0x66, 0x54, 0x4e, 0x44, 0x50,
		0x4b, 0x32, 0x48, 0x53, 0x79, 0x48, 0x64, 0x53, 0x75, 0x63,
		0x6a, 0x59, 0x31, 0x71, 0x73, 0x36, 0x57, 0x41, 0x75, 0x62,
		0x36, 0x6f, 0x57, 0x48, 0x4a, 0x75, 0x4c, 0x42, 0x78, 0x4d,
		0x65, 0x73, 0x66, 0x74, 0x70, 0x6e, 0x55, 0x77, 0x6f, 0x4c,
		0x6e, 0x56, 0x5a, 0x79, 0x4e, 0x2b, 0x70, 0x4f, 0x62, 0x6c,
		0x55, 0x5a, 0x0a, 0x55, 0x4d, 0x73, 0x76, 0x78, 0x50, 0x33,
		0x50, 0x6c, 0x53, 0x2b, 0x6d, 0x41, 0x2b, 0x45, 0x36, 0x4e,
		0x79, 0x51, 0x58, 0x30, 0x46, 0x2f, 0x66, 0x63, 0x66, 0x47,
		0x4c, 0x41, 0x67, 0x4d, 0x42, 0x41, 0x41, 0x45, 0x43, 0x67,
		0x59, 0x41, 0x30, 0x59, 0x4e, 0x4b, 0x68, 0x55, 0x45, 0x64,
		0x38, 0x78, 0x74, 0x47, 0x67, 0x44, 0x44, 0x79, 0x67, 0x2f,
		0x46, 0x47, 0x54, 0x55, 0x50, 0x64, 0x34, 0x0a, 0x42, 0x6b,
		0x6b, 0x49, 0x42, 0x31, 0x58, 0x55, 0x56, 0x4a, 0x79, 0x57,
		0x5a, 0x36, 0x50, 0x51, 0x47, 0x51, 0x59, 0x75, 0x33, 0x73,
		0x2b, 0x65, 0x39, 0x52, 0x58, 0x37, 0x67, 0x50, 0x78, 0x52,
		0x41, 0x63, 0x58, 0x61, 0x32, 0x7a, 0x63, 0x54, 0x37, 0x39,
		0x37, 0x6a, 0x79, 0x4f, 0x6c, 0x73, 0x7a, 0x52, 0x57, 0x68,
		0x53, 0x50, 0x6e, 0x65, 0x4e, 0x74, 0x70, 0x62, 0x48, 0x56,
		0x51, 0x74, 0x0a, 0x57, 0x5a, 0x44, 0x56, 0x79, 0x6b, 0x6c,
		0x6a, 0x52, 0x63, 0x75, 0x78, 0x46, 0x53, 0x5a, 0x70, 0x42,
		0x30, 0x37, 0x41, 0x53, 0x64, 0x4b, 0x42, 0x65, 0x4f, 0x31,
		0x44, 0x4c, 0x61, 0x47, 0x32, 0x57, 0x32, 0x48, 0x78, 0x47,
		0x4b, 0x63, 0x48, 0x34, 0x75, 0x64, 0x55, 0x62, 0x78, 0x35,
		0x52, 0x4c, 0x32, 0x4e, 0x61, 0x63, 0x51, 0x53, 0x4b, 0x52,
		0x2b, 0x38, 0x49, 0x73, 0x66, 0x53, 0x6d, 0x0a, 0x32, 0x51,
		0x35, 0x72, 0x2b, 0x37, 0x6f, 0x59, 0x6f, 0x56, 0x51, 0x79,
		0x50, 0x35, 0x44, 0x33, 0x6d, 0x51, 0x4a, 0x42, 0x41, 0x4f,
		0x47, 0x57, 0x59, 0x2f, 0x32, 0x41, 0x2b, 0x63, 0x62, 0x34,
		0x38, 0x52, 0x4c, 0x68, 0x4e, 0x4c, 0x6b, 0x64, 0x43, 0x6e,
		0x6e, 0x62, 0x47, 0x58, 0x68, 0x6b, 0x2b, 0x6f, 0x68, 0x36,
		0x73, 0x4c, 0x5a, 0x69, 0x62, 0x69, 0x7a, 0x62, 0x6f, 0x52,
		0x57, 0x6e, 0x0a, 0x6c, 0x79, 0x4d, 0x4f, 0x62, 0x31, 0x31,
		0x32, 0x5a, 0x6f, 0x50, 0x50, 0x46, 0x50, 0x35, 0x53, 0x4b,
		0x4e, 0x72, 0x64, 0x41, 0x78, 0x45, 0x6d, 0x45, 0x32, 0x33,
		0x31, 0x34, 0x4a, 0x63, 0x4e, 0x48, 0x72, 0x75, 0x74, 0x6a,
		0x4d, 0x46, 0x6d, 0x2f, 0x75, 0x63, 0x43, 0x51, 0x51, 0x44,
		0x42, 0x73, 0x78, 0x54, 0x54, 0x41, 0x7a, 0x6f, 0x78, 0x68,
		0x6f, 0x4f, 0x7a, 0x67, 0x56, 0x38, 0x52, 0x0a, 0x36, 0x35,
		0x4e, 0x64, 0x50, 0x58, 0x67, 0x2f, 0x64, 0x78, 0x64, 0x6b,
		0x34, 0x55, 0x61, 0x4b, 0x51, 0x6b, 0x39, 0x61, 0x49, 0x68,
		0x51, 0x61, 0x67, 0x77, 0x61, 0x38, 0x4f, 0x42, 0x6a, 0x50,
		0x46, 0x32, 0x49, 0x57, 0x67, 0x66, 0x4d, 0x5a, 0x30, 0x75,
		0x52, 0x71, 0x6d, 0x77, 0x2b, 0x4c, 0x4e, 0x4f, 0x6f, 0x45,
		0x6f, 0x79, 0x31, 0x68, 0x44, 0x30, 0x4a, 0x68, 0x42, 0x72,
		0x79, 0x44, 0x0a, 0x56, 0x78, 0x65, 0x39, 0x41, 0x6b, 0x45,
		0x41, 0x79, 0x77, 0x4b, 0x75, 0x5a, 0x56, 0x71, 0x47, 0x62,
		0x64, 0x74, 0x6d, 0x42, 0x39, 0x6d, 0x48, 0x75, 0x76, 0x63,
		0x35, 0x6b, 0x45, 0x50, 0x75, 0x66, 0x66, 0x78, 0x52, 0x77,
		0x6a, 0x53, 0x33, 0x68, 0x73, 0x71, 0x35, 0x33, 0x38, 0x43,
		0x66, 0x44, 0x48, 0x2f, 0x50, 0x63, 0x59, 0x72, 0x79, 0x43,
		0x61, 0x67, 0x64, 0x78, 0x59, 0x79, 0x38, 0x0a, 0x6c, 0x63,
		0x71, 0x57, 0x58, 0x61, 0x2f, 0x37, 0x72, 0x4a, 0x6b, 0x5a,
		0x62, 0x79, 0x47, 0x51, 0x78, 0x68, 0x37, 0x57, 0x67, 0x34,
		0x74, 0x42, 0x57, 0x6d, 0x4d, 0x34, 0x44, 0x51, 0x4a, 0x41,
		0x4c, 0x38, 0x4d, 0x59, 0x76, 0x32, 0x39, 0x73, 0x53, 0x67,
		0x6f, 0x42, 0x4c, 0x36, 0x49, 0x57, 0x37, 0x7a, 0x52, 0x48,
		0x67, 0x68, 0x5a, 0x47, 0x4d, 0x47, 0x41, 0x4e, 0x52, 0x4c,
		0x4c, 0x48, 0x0a, 0x30, 0x67, 0x2f, 0x48, 0x77, 0x56, 0x48,
		0x6c, 0x34, 0x79, 0x4f, 0x72, 0x35, 0x58, 0x31, 0x76, 0x6f,
		0x4b, 0x45, 0x44, 0x62, 0x73, 0x6c, 0x63, 0x53, 0x47, 0x48,
		0x59, 0x4d, 0x50, 0x46, 0x4c, 0x51, 0x2b, 0x67, 0x6f, 0x54,
		0x44, 0x78, 0x77, 0x56, 0x42, 0x36, 0x50, 0x48, 0x35, 0x32,
		0x70, 0x6e, 0x6a, 0x6b, 0x37, 0x67, 0x51, 0x4a, 0x41, 0x55,
		0x53, 0x48, 0x69, 0x56, 0x2f, 0x55, 0x51, 0x0a, 0x6b, 0x54,
		0x7a, 0x33, 0x42, 0x6c, 0x78, 0x5a, 0x63, 0x31, 0x49, 0x41,
		0x69, 0x55, 0x51, 0x76, 0x39, 0x2f, 0x42, 0x61, 0x38, 0x77,
		0x74, 0x48, 0x57, 0x53, 0x56, 0x70, 0x2b, 0x59, 0x71, 0x50,
		0x68, 0x78, 0x74, 0x31, 0x73, 0x66, 0x64, 0x69, 0x79, 0x55,
		0x4d, 0x58, 0x74, 0x6c, 0x41, 0x34, 0x66, 0x36, 0x57, 0x41,
		0x4b, 0x41, 0x47, 0x4d, 0x72, 0x61, 0x6f, 0x52, 0x77, 0x34,
		0x77, 0x49, 0x0a, 0x63, 0x59, 0x72, 0x2b, 0x4e, 0x36, 0x57,
		0x78, 0x2b, 0x77, 0x77, 0x58, 0x5a, 0x77, 0x3d, 0x3d, 0x0a,
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x50,
		0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59,
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a,
	}},
};
/*

RSASHA256 (2048b)

rsa2048-256.	DNSKEY	  256  3  8  AwEAAb0vt7wVuqSPgSU1qrMiriq/ay57oVEzb/w7OKmi48bRjvK9x178cfbS03wUWqY9mwD72rUZ4urwTDaibkqBiTdjQOejiOmcfvWdqLpR35Jh8CzpfgZQHrCOeUFXKLVXY75h0oN1tqVJzRHzi36TB99WtM8P4MayKjlWMoaeBwLUpx6EfVy/oir1dFAxkGJ7ko6Pojhj5CNh0tprt32yp+VTo8bkKukbo+ho1ouqQ7EzpeimrVh3H0udgYq/fMF3NdSSEmr+/xglwOAoKimpmlKsYTwdNcL2Du6kZj50gC6hMqgJy9pB+BRyqJCWlIum6VqyB/SfMVs0rVVCf17+3BM=
rsa2048-256.	DS    	53905  8  1  ACA228FB117080FB5AD9641D26A5326405D4BF76
rsa2048-256.	DS    	53905  8  2  32ABCBB109B03FE2370A331B94E11A54FC336EFA7E94304926DAB46609452087
rsa2048-256.	DS    	53905  8  4  1422FBE78066402142E98167604CF4AE5F7AAA5CAE3DA77C9941BA137B7B7E18DCD776C3BC5F6C31816970DBE8E0088B

-----BEGIN PRIVATE KEY-----
MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQC9L7e8Fbqkj4El
NaqzIq4qv2sue6FRM2/8OzipouPG0Y7yvcde/HH20tN8FFqmPZsA+9q1GeLq8Ew2
om5KgYk3Y0Dno4jpnH71nai6Ud+SYfAs6X4GUB6wjnlBVyi1V2O+YdKDdbalSc0R
84t+kwffVrTPD+DGsio5VjKGngcC1KcehH1cv6Iq9XRQMZBie5KOj6I4Y+QjYdLa
a7d9sqflU6PG5CrpG6PoaNaLqkOxM6Xopq1Ydx9LnYGKv3zBdzXUkhJq/v8YJcDg
KCopqZpSrGE8HTXC9g7upGY+dIAuoTKoCcvaQfgUcqiQlpSLpulasgf0nzFbNK1V
Qn9e/twTAgMBAAECggEAW6W2GENee0GZinEJFGaEgOjh/06cwz1rzoDR3juupC34
Vddhsb4RZ0SSdtVVZJsLb9qSBJFZv/5QrVbanvVJh69pTc2HEcRiLJIHntryt3MC
an9iMaS2eUlLrD+Q1OhUxXGWDWEE7eVmln+o39RSR/s08+n5NcgxdD0xlM0REGkm
hspOIF3ZXgv+7gH2It7sjjfojNgbZ2psCABWxTwHof1ea4ZQlvYC+aCpgG3slsqs
veeMMIWiQBK/zHaBFi1NE17lz9HdQgjCOPm/KsLBwBwIsXT2iaUWNha06lU2qjK4
0iv72C8O6S3z9anNO8RzWCUpKixxtAFMtnVDgpK5oQKBgQDjU0jU1D2LkrFdsfYy
TfxHO1IUP8Eccpjg+Clph3aBMe5UiUSKpDDAWnMZ08zRufBjFSGNMzAqSz+nVN8l
UcKT1fditME/Ty3NVkTXUnxbQ5VqguJSdtbJThaUyxnwJecKOde051hDCIm78Kok
RYuNDP3Y5cbOPRRQbfGP7REdjQKBgQDVDNz8RnwCseWbx1k7yZAJ56MqI5fHiHwn
DRbeSStvxaS3tEezYrMCZGtwZX4w3xxFDA8XXzkQwiP9X6iw3wbzdnXXv9sCz0Yq
SrwiQw0FPJXw+Gb6qRTKZCrGL4GlfNzWwZDV7GgNrbihghtQAdgFeJC54nNmoTb3
mIDyF1doHwKBgQDecQOmhtyGdA+s0FRJ2zfOeASUHeKYNKbWFN9pDQ3v0+0TuZJh
W9L8lkVk83TLkHfibK39Hia2bBWoZotIWF3fDZNaeedTTOZTtGP+kw9uuo49wD84
VGNObmP4S1lhPaV7wygJNAuQ5IzqY9AJWCf66MiI8ib8bpYvMbYbkO2bjQKBgEzw
3E3vVpHSlU1w7PT+khpBoVdyiznwPIYN4Bhe7GTtR9NgNBTArE1M3tA0lyInMRpu
hHS4Wu+GbWDvzwgxQ11hhqVoEDwvhvN3EgJ9nJIrkyr3SplzvqJpElmIhpWxRj/4
n8niGqH1/xu01RBL2iv/oqS+91Y1GpjIZqkHbPWZAoGBAIaOVlPKAwyo+gm/QXhu
d++Kw8Us5tY3TSvCvyF8Ht7nskwxXFhxqTBtE2UETnvs1vo39FDSSW60JnmQ2fVO
PJ7z/AxmR0ZOKK9iNJhfVhImhvRW7M/S3+HnfO2MkeQnuzM/S4ECRv9d0pUJQZKC
PXo7nlETmd8tfA0ltYZEj3WW
-----END PRIVATE KEY-----

*/
static const key_parameters_t SAMPLE_RSA2048_SHA256_KEY = {
	.name = (uint8_t *)"\x0b""rsa2048-256",
	.flags = 256,
	.protocol = 3,
	.algorithm = 8,
	.public_key = { .size = 260, .data = (uint8_t []){
		0x03, 0x01, 0x00, 0x01, 0xbd, 0x2f, 0xb7, 0xbc, 0x15, 0xba,
		0xa4, 0x8f, 0x81, 0x25, 0x35, 0xaa, 0xb3, 0x22, 0xae, 0x2a,
		0xbf, 0x6b, 0x2e, 0x7b, 0xa1, 0x51, 0x33, 0x6f, 0xfc, 0x3b,
		0x38, 0xa9, 0xa2, 0xe3, 0xc6, 0xd1, 0x8e, 0xf2, 0xbd, 0xc7,
		0x5e, 0xfc, 0x71, 0xf6, 0xd2, 0xd3, 0x7c, 0x14, 0x5a, 0xa6,
		0x3d, 0x9b, 0x00, 0xfb, 0xda, 0xb5, 0x19, 0xe2, 0xea, 0xf0,
		0x4c, 0x36, 0xa2, 0x6e, 0x4a, 0x81, 0x89, 0x37, 0x63, 0x40,
		0xe7, 0xa3, 0x88, 0xe9, 0x9c, 0x7e, 0xf5, 0x9d, 0xa8, 0xba,
		0x51, 0xdf, 0x92, 0x61, 0xf0, 0x2c, 0xe9, 0x7e, 0x06, 0x50,
		0x1e, 0xb0, 0x8e, 0x79, 0x41, 0x57, 0x28, 0xb5, 0x57, 0x63,
		0xbe, 0x61, 0xd2, 0x83, 0x75, 0xb6, 0xa5, 0x49, 0xcd, 0x11,
		0xf3, 0x8b, 0x7e, 0x93, 0x07, 0xdf, 0x56, 0xb4, 0xcf, 0x0f,
		0xe0, 0xc6, 0xb2, 0x2a, 0x39, 0x56, 0x32, 0x86, 0x9e, 0x07,
		0x02, 0xd4, 0xa7, 0x1e, 0x84, 0x7d, 0x5c, 0xbf, 0xa2, 0x2a,
		0xf5, 0x74, 0x50, 0x31, 0x90, 0x62, 0x7b, 0x92, 0x8e, 0x8f,
		0xa2, 0x38, 0x63, 0xe4, 0x23, 0x61, 0xd2, 0xda, 0x6b, 0xb7,
		0x7d, 0xb2, 0xa7, 0xe5, 0x53, 0xa3, 0xc6, 0xe4, 0x2a, 0xe9,
		0x1b, 0xa3, 0xe8, 0x68, 0xd6, 0x8b, 0xaa, 0x43, 0xb1, 0x33,
		0xa5, 0xe8, 0xa6, 0xad, 0x58, 0x77, 0x1f, 0x4b, 0x9d, 0x81,
		0x8a, 0xbf, 0x7c, 0xc1, 0x77, 0x35, 0xd4, 0x92, 0x12, 0x6a,
		0xfe, 0xff, 0x18, 0x25, 0xc0, 0xe0, 0x28, 0x2a, 0x29, 0xa9,
		0x9a, 0x52, 0xac, 0x61, 0x3c, 0x1d, 0x35, 0xc2, 0xf6, 0x0e,
		0xee, 0xa4, 0x66, 0x3e, 0x74, 0x80, 0x2e, 0xa1, 0x32, 0xa8,
		0x09, 0xcb, 0xda, 0x41, 0xf8, 0x14, 0x72, 0xa8, 0x90, 0x96,
		0x94, 0x8b, 0xa6, 0xe9, 0x5a, 0xb2, 0x07, 0xf4, 0x9f, 0x31,
		0x5b, 0x34, 0xad, 0x55, 0x42, 0x7f, 0x5e, 0xfe, 0xdc, 0x13,
	}},
	.rdata = { .size = 264, .data = (uint8_t []){
		01, 0x00, 0x03, 0x08,
		0x03, 0x01, 0x00, 0x01, 0xbd, 0x2f, 0xb7, 0xbc, 0x15, 0xba,
		0xa4, 0x8f, 0x81, 0x25, 0x35, 0xaa, 0xb3, 0x22, 0xae, 0x2a,
		0xbf, 0x6b, 0x2e, 0x7b, 0xa1, 0x51, 0x33, 0x6f, 0xfc, 0x3b,
		0x38, 0xa9, 0xa2, 0xe3, 0xc6, 0xd1, 0x8e, 0xf2, 0xbd, 0xc7,
		0x5e, 0xfc, 0x71, 0xf6, 0xd2, 0xd3, 0x7c, 0x14, 0x5a, 0xa6,
		0x3d, 0x9b, 0x00, 0xfb, 0xda, 0xb5, 0x19, 0xe2, 0xea, 0xf0,
		0x4c, 0x36, 0xa2, 0x6e, 0x4a, 0x81, 0x89, 0x37, 0x63, 0x40,
		0xe7, 0xa3, 0x88, 0xe9, 0x9c, 0x7e, 0xf5, 0x9d, 0xa8, 0xba,
		0x51, 0xdf, 0x92, 0x61, 0xf0, 0x2c, 0xe9, 0x7e, 0x06, 0x50,
		0x1e, 0xb0, 0x8e, 0x79, 0x41, 0x57, 0x28, 0xb5, 0x57, 0x63,
		0xbe, 0x61, 0xd2, 0x83, 0x75, 0xb6, 0xa5, 0x49, 0xcd, 0x11,
		0xf3, 0x8b, 0x7e, 0x93, 0x07, 0xdf, 0x56, 0xb4, 0xcf, 0x0f,
		0xe0, 0xc6, 0xb2, 0x2a, 0x39, 0x56, 0x32, 0x86, 0x9e, 0x07,
		0x02, 0xd4, 0xa7, 0x1e, 0x84, 0x7d, 0x5c, 0xbf, 0xa2, 0x2a,
		0xf5, 0x74, 0x50, 0x31, 0x90, 0x62, 0x7b, 0x92, 0x8e, 0x8f,
		0xa2, 0x38, 0x63, 0xe4, 0x23, 0x61, 0xd2, 0xda, 0x6b, 0xb7,
		0x7d, 0xb2, 0xa7, 0xe5, 0x53, 0xa3, 0xc6, 0xe4, 0x2a, 0xe9,
		0x1b, 0xa3, 0xe8, 0x68, 0xd6, 0x8b, 0xaa, 0x43, 0xb1, 0x33,
		0xa5, 0xe8, 0xa6, 0xad, 0x58, 0x77, 0x1f, 0x4b, 0x9d, 0x81,
		0x8a, 0xbf, 0x7c, 0xc1, 0x77, 0x35, 0xd4, 0x92, 0x12, 0x6a,
		0xfe, 0xff, 0x18, 0x25, 0xc0, 0xe0, 0x28, 0x2a, 0x29, 0xa9,
		0x9a, 0x52, 0xac, 0x61, 0x3c, 0x1d, 0x35, 0xc2, 0xf6, 0x0e,
		0xee, 0xa4, 0x66, 0x3e, 0x74, 0x80, 0x2e, 0xa1, 0x32, 0xa8,
		0x09, 0xcb, 0xda, 0x41, 0xf8, 0x14, 0x72, 0xa8, 0x90, 0x96,
		0x94, 0x8b, 0xa6, 0xe9, 0x5a, 0xb2, 0x07, 0xf4, 0x9f, 0x31,
		0x5b, 0x34, 0xad, 0x55, 0x42, 0x7f, 0x5e, 0xfe, 0xdc, 0x13,
	}},
	.key_id = "712dc810fb689f35e9886d2d155d49bc3e59518e",
	.keytag = 53905,
	.ds_sha1 = { .size = 24, .data = (uint8_t []){
		0xd2, 0x91, 0x08, 0x01, 0xac, 0xa2, 0x28, 0xfb, 0x11, 0x70,
		0x80, 0xfb, 0x5a, 0xd9, 0x64, 0x1d, 0x26, 0xa5, 0x32, 0x64,
		0x05, 0xd4, 0xbf, 0x76,
	}},
	.ds_sha256 = { .size = 36, .data = (uint8_t []){
		0xd2, 0x91, 0x08, 0x02, 0x32, 0xab, 0xcb, 0xb1, 0x09, 0xb0,
		0x3f, 0xe2, 0x37, 0x0a, 0x33, 0x1b, 0x94, 0xe1, 0x1a, 0x54,
		0xfc, 0x33, 0x6e, 0xfa, 0x7e, 0x94, 0x30, 0x49, 0x26, 0xda,
		0xb4, 0x66, 0x09, 0x45, 0x20, 0x87,
	}},
	.ds_sha384 = { .size = 52, .data = (uint8_t []){
		0xd2, 0x91, 0x08, 0x04, 0x14, 0x22, 0xfb, 0xe7, 0x80, 0x66,
		0x40, 0x21, 0x42, 0xe9, 0x81, 0x67, 0x60, 0x4c, 0xf4, 0xae,
		0x5f, 0x7a, 0xaa, 0x5c, 0xae, 0x3d, 0xa7, 0x7c, 0x99, 0x41,
		0xba, 0x13, 0x7b, 0x7b, 0x7e, 0x18, 0xdc, 0xd7, 0x76, 0xc3,
		0xbc, 0x5f, 0x6c, 0x31, 0x81, 0x69, 0x70, 0xdb, 0xe8, 0xe0,
		0x08, 0x8b,
	}},
	.bit_size = 2048,
	.pem = { .size = 1704, .data = (uint8_t []){
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49,
		0x49, 0x45, 0x76, 0x67, 0x49, 0x42, 0x41, 0x44, 0x41, 0x4e,
		0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77,
		0x30, 0x42, 0x41, 0x51, 0x45, 0x46, 0x41, 0x41, 0x53, 0x43,
		0x42, 0x4b, 0x67, 0x77, 0x67, 0x67, 0x53, 0x6b, 0x41, 0x67,
		0x45, 0x41, 0x41, 0x6f, 0x49, 0x42, 0x41, 0x51, 0x43, 0x39,
		0x4c, 0x37, 0x65, 0x38, 0x46, 0x62, 0x71, 0x6b, 0x6a, 0x34,
		0x45, 0x6c, 0x0a, 0x4e, 0x61, 0x71, 0x7a, 0x49, 0x71, 0x34,
		0x71, 0x76, 0x32, 0x73, 0x75, 0x65, 0x36, 0x46, 0x52, 0x4d,
		0x32, 0x2f, 0x38, 0x4f, 0x7a, 0x69, 0x70, 0x6f, 0x75, 0x50,
		0x47, 0x30, 0x59, 0x37, 0x79, 0x76, 0x63, 0x64, 0x65, 0x2f,
		0x48, 0x48, 0x32, 0x30, 0x74, 0x4e, 0x38, 0x46, 0x46, 0x71,
		0x6d, 0x50, 0x5a, 0x73, 0x41, 0x2b, 0x39, 0x71, 0x31, 0x47,
		0x65, 0x4c, 0x71, 0x38, 0x45, 0x77, 0x32, 0x0a, 0x6f, 0x6d,
		0x35, 0x4b, 0x67, 0x59, 0x6b, 0x33, 0x59, 0x30, 0x44, 0x6e,
		0x6f, 0x34, 0x6a, 0x70, 0x6e, 0x48, 0x37, 0x31, 0x6e, 0x61,
		0x69, 0x36, 0x55, 0x64, 0x2b, 0x53, 0x59, 0x66, 0x41, 0x73,
		0x36, 0x58, 0x34, 0x47, 0x55, 0x42, 0x36, 0x77, 0x6a, 0x6e,
		0x6c, 0x42, 0x56, 0x79, 0x69, 0x31, 0x56, 0x32, 0x4f, 0x2b,
		0x59, 0x64, 0x4b, 0x44, 0x64, 0x62, 0x61, 0x6c, 0x53, 0x63,
		0x30, 0x52, 0x0a, 0x38, 0x34, 0x74, 0x2b, 0x6b, 0x77, 0x66,
		0x66, 0x56, 0x72, 0x54, 0x50, 0x44, 0x2b, 0x44, 0x47, 0x73,
		0x69, 0x6f, 0x35, 0x56, 0x6a, 0x4b, 0x47, 0x6e, 0x67, 0x63,
		0x43, 0x31, 0x4b, 0x63, 0x65, 0x68, 0x48, 0x31, 0x63, 0x76,
		0x36, 0x49, 0x71, 0x39, 0x58, 0x52, 0x51, 0x4d, 0x5a, 0x42,
		0x69, 0x65, 0x35, 0x4b, 0x4f, 0x6a, 0x36, 0x49, 0x34, 0x59,
		0x2b, 0x51, 0x6a, 0x59, 0x64, 0x4c, 0x61, 0x0a, 0x61, 0x37,
		0x64, 0x39, 0x73, 0x71, 0x66, 0x6c, 0x55, 0x36, 0x50, 0x47,
		0x35, 0x43, 0x72, 0x70, 0x47, 0x36, 0x50, 0x6f, 0x61, 0x4e,
		0x61, 0x4c, 0x71, 0x6b, 0x4f, 0x78, 0x4d, 0x36, 0x58, 0x6f,
		0x70, 0x71, 0x31, 0x59, 0x64, 0x78, 0x39, 0x4c, 0x6e, 0x59,
		0x47, 0x4b, 0x76, 0x33, 0x7a, 0x42, 0x64, 0x7a, 0x58, 0x55,
		0x6b, 0x68, 0x4a, 0x71, 0x2f, 0x76, 0x38, 0x59, 0x4a, 0x63,
		0x44, 0x67, 0x0a, 0x4b, 0x43, 0x6f, 0x70, 0x71, 0x5a, 0x70,
		0x53, 0x72, 0x47, 0x45, 0x38, 0x48, 0x54, 0x58, 0x43, 0x39,
		0x67, 0x37, 0x75, 0x70, 0x47, 0x59, 0x2b, 0x64, 0x49, 0x41,
		0x75, 0x6f, 0x54, 0x4b, 0x6f, 0x43, 0x63, 0x76, 0x61, 0x51,
		0x66, 0x67, 0x55, 0x63, 0x71, 0x69, 0x51, 0x6c, 0x70, 0x53,
		0x4c, 0x70, 0x75, 0x6c, 0x61, 0x73, 0x67, 0x66, 0x30, 0x6e,
		0x7a, 0x46, 0x62, 0x4e, 0x4b, 0x31, 0x56, 0x0a, 0x51, 0x6e,
		0x39, 0x65, 0x2f, 0x74, 0x77, 0x54, 0x41, 0x67, 0x4d, 0x42,
		0x41, 0x41, 0x45, 0x43, 0x67, 0x67, 0x45, 0x41, 0x57, 0x36,
		0x57, 0x32, 0x47, 0x45, 0x4e, 0x65, 0x65, 0x30, 0x47, 0x5a,
		0x69, 0x6e, 0x45, 0x4a, 0x46, 0x47, 0x61, 0x45, 0x67, 0x4f,
		0x6a, 0x68, 0x2f, 0x30, 0x36, 0x63, 0x77, 0x7a, 0x31, 0x72,
		0x7a, 0x6f, 0x44, 0x52, 0x33, 0x6a, 0x75, 0x75, 0x70, 0x43,
		0x33, 0x34, 0x0a, 0x56, 0x64, 0x64, 0x68, 0x73, 0x62, 0x34,
		0x52, 0x5a, 0x30, 0x53, 0x53, 0x64, 0x74, 0x56, 0x56, 0x5a,
		0x4a, 0x73, 0x4c, 0x62, 0x39, 0x71, 0x53, 0x42, 0x4a, 0x46,
		0x5a, 0x76, 0x2f, 0x35, 0x51, 0x72, 0x56, 0x62, 0x61, 0x6e,
		0x76, 0x56, 0x4a, 0x68, 0x36, 0x39, 0x70, 0x54, 0x63, 0x32,
		0x48, 0x45, 0x63, 0x52, 0x69, 0x4c, 0x4a, 0x49, 0x48, 0x6e,
		0x74, 0x72, 0x79, 0x74, 0x33, 0x4d, 0x43, 0x0a, 0x61, 0x6e,
		0x39, 0x69, 0x4d, 0x61, 0x53, 0x32, 0x65, 0x55, 0x6c, 0x4c,
		0x72, 0x44, 0x2b, 0x51, 0x31, 0x4f, 0x68, 0x55, 0x78, 0x58,
		0x47, 0x57, 0x44, 0x57, 0x45, 0x45, 0x37, 0x65, 0x56, 0x6d,
		0x6c, 0x6e, 0x2b, 0x6f, 0x33, 0x39, 0x52, 0x53, 0x52, 0x2f,
		0x73, 0x30, 0x38, 0x2b, 0x6e, 0x35, 0x4e, 0x63, 0x67, 0x78,
		0x64, 0x44, 0x30, 0x78, 0x6c, 0x4d, 0x30, 0x52, 0x45, 0x47,
		0x6b, 0x6d, 0x0a, 0x68, 0x73, 0x70, 0x4f, 0x49, 0x46, 0x33,
		0x5a, 0x58, 0x67, 0x76, 0x2b, 0x37, 0x67, 0x48, 0x32, 0x49,
		0x74, 0x37, 0x73, 0x6a, 0x6a, 0x66, 0x6f, 0x6a, 0x4e, 0x67,
		0x62, 0x5a, 0x32, 0x70, 0x73, 0x43, 0x41, 0x42, 0x57, 0x78,
		0x54, 0x77, 0x48, 0x6f, 0x66, 0x31, 0x65, 0x61, 0x34, 0x5a,
		0x51, 0x6c, 0x76, 0x59, 0x43, 0x2b, 0x61, 0x43, 0x70, 0x67,
		0x47, 0x33, 0x73, 0x6c, 0x73, 0x71, 0x73, 0x0a, 0x76, 0x65,
		0x65, 0x4d, 0x4d, 0x49, 0x57, 0x69, 0x51, 0x42, 0x4b, 0x2f,
		0x7a, 0x48, 0x61, 0x42, 0x46, 0x69, 0x31, 0x4e, 0x45, 0x31,
		0x37, 0x6c, 0x7a, 0x39, 0x48, 0x64, 0x51, 0x67, 0x6a, 0x43,
		0x4f, 0x50, 0x6d, 0x2f, 0x4b, 0x73, 0x4c, 0x42, 0x77, 0x42,
		0x77, 0x49, 0x73, 0x58, 0x54, 0x32, 0x69, 0x61, 0x55, 0x57,
		0x4e, 0x68, 0x61, 0x30, 0x36, 0x6c, 0x55, 0x32, 0x71, 0x6a,
		0x4b, 0x34, 0x0a, 0x30, 0x69, 0x76, 0x37, 0x32, 0x43, 0x38,
		0x4f, 0x36, 0x53, 0x33, 0x7a, 0x39, 0x61, 0x6e, 0x4e, 0x4f,
		0x38, 0x52, 0x7a, 0x57, 0x43, 0x55, 0x70, 0x4b, 0x69, 0x78,
		0x78, 0x74, 0x41, 0x46, 0x4d, 0x74, 0x6e, 0x56, 0x44, 0x67,
		0x70, 0x4b, 0x35, 0x6f, 0x51, 0x4b, 0x42, 0x67, 0x51, 0x44,
		0x6a, 0x55, 0x30, 0x6a, 0x55, 0x31, 0x44, 0x32, 0x4c, 0x6b,
		0x72, 0x46, 0x64, 0x73, 0x66, 0x59, 0x79, 0x0a, 0x54, 0x66,
		0x78, 0x48, 0x4f, 0x31, 0x49, 0x55, 0x50, 0x38, 0x45, 0x63,
		0x63, 0x70, 0x6a, 0x67, 0x2b, 0x43, 0x6c, 0x70, 0x68, 0x33,
		0x61, 0x42, 0x4d, 0x65, 0x35, 0x55, 0x69, 0x55, 0x53, 0x4b,
		0x70, 0x44, 0x44, 0x41, 0x57, 0x6e, 0x4d, 0x5a, 0x30, 0x38,
		0x7a, 0x52, 0x75, 0x66, 0x42, 0x6a, 0x46, 0x53, 0x47, 0x4e,
		0x4d, 0x7a, 0x41, 0x71, 0x53, 0x7a, 0x2b, 0x6e, 0x56, 0x4e,
		0x38, 0x6c, 0x0a, 0x55, 0x63, 0x4b, 0x54, 0x31, 0x66, 0x64,
		0x69, 0x74, 0x4d, 0x45, 0x2f, 0x54, 0x79, 0x33, 0x4e, 0x56,
		0x6b, 0x54, 0x58, 0x55, 0x6e, 0x78, 0x62, 0x51, 0x35, 0x56,
		0x71, 0x67, 0x75, 0x4a, 0x53, 0x64, 0x74, 0x62, 0x4a, 0x54,
		0x68, 0x61, 0x55, 0x79, 0x78, 0x6e, 0x77, 0x4a, 0x65, 0x63,
		0x4b, 0x4f, 0x64, 0x65, 0x30, 0x35, 0x31, 0x68, 0x44, 0x43,
		0x49, 0x6d, 0x37, 0x38, 0x4b, 0x6f, 0x6b, 0x0a, 0x52, 0x59,
		0x75, 0x4e, 0x44, 0x50, 0x33, 0x59, 0x35, 0x63, 0x62, 0x4f,
		0x50, 0x52, 0x52, 0x51, 0x62, 0x66, 0x47, 0x50, 0x37, 0x52,
		0x45, 0x64, 0x6a, 0x51, 0x4b, 0x42, 0x67, 0x51, 0x44, 0x56,
		0x44, 0x4e, 0x7a, 0x38, 0x52, 0x6e, 0x77, 0x43, 0x73, 0x65,
		0x57, 0x62, 0x78, 0x31, 0x6b, 0x37, 0x79, 0x5a, 0x41, 0x4a,
		0x35, 0x36, 0x4d, 0x71, 0x49, 0x35, 0x66, 0x48, 0x69, 0x48,
		0x77, 0x6e, 0x0a, 0x44, 0x52, 0x62, 0x65, 0x53, 0x53, 0x74,
		0x76, 0x78, 0x61, 0x53, 0x33, 0x74, 0x45, 0x65, 0x7a, 0x59,
		0x72, 0x4d, 0x43, 0x5a, 0x47, 0x74, 0x77, 0x5a, 0x58, 0x34,
		0x77, 0x33, 0x78, 0x78, 0x46, 0x44, 0x41, 0x38, 0x58, 0x58,
		0x7a, 0x6b, 0x51, 0x77, 0x69, 0x50, 0x39, 0x58, 0x36, 0x69,
		0x77, 0x33, 0x77, 0x62, 0x7a, 0x64, 0x6e, 0x58, 0x58, 0x76,
		0x39, 0x73, 0x43, 0x7a, 0x30, 0x59, 0x71, 0x0a, 0x53, 0x72,
		0x77, 0x69, 0x51, 0x77, 0x30, 0x46, 0x50, 0x4a, 0x58, 0x77,
		0x2b, 0x47, 0x62, 0x36, 0x71, 0x52, 0x54, 0x4b, 0x5a, 0x43,
		0x72, 0x47, 0x4c, 0x34, 0x47, 0x6c, 0x66, 0x4e, 0x7a, 0x57,
		0x77, 0x5a, 0x44, 0x56, 0x37, 0x47, 0x67, 0x4e, 0x72, 0x62,
		0x69, 0x68, 0x67, 0x68, 0x74, 0x51, 0x41, 0x64, 0x67, 0x46,
		0x65, 0x4a, 0x43, 0x35, 0x34, 0x6e, 0x4e, 0x6d, 0x6f, 0x54,
		0x62, 0x33, 0x0a, 0x6d, 0x49, 0x44, 0x79, 0x46, 0x31, 0x64,
		0x6f, 0x48, 0x77, 0x4b, 0x42, 0x67, 0x51, 0x44, 0x65, 0x63,
		0x51, 0x4f, 0x6d, 0x68, 0x74, 0x79, 0x47, 0x64, 0x41, 0x2b,
		0x73, 0x30, 0x46, 0x52, 0x4a, 0x32, 0x7a, 0x66, 0x4f, 0x65,
		0x41, 0x53, 0x55, 0x48, 0x65, 0x4b, 0x59, 0x4e, 0x4b, 0x62,
		0x57, 0x46, 0x4e, 0x39, 0x70, 0x44, 0x51, 0x33, 0x76, 0x30,
		0x2b, 0x30, 0x54, 0x75, 0x5a, 0x4a, 0x68, 0x0a, 0x57, 0x39,
		0x4c, 0x38, 0x6c, 0x6b, 0x56, 0x6b, 0x38, 0x33, 0x54, 0x4c,
		0x6b, 0x48, 0x66, 0x69, 0x62, 0x4b, 0x33, 0x39, 0x48, 0x69,
		0x61, 0x32, 0x62, 0x42, 0x57, 0x6f, 0x5a, 0x6f, 0x74, 0x49,
		0x57, 0x46, 0x33, 0x66, 0x44, 0x5a, 0x4e, 0x61, 0x65, 0x65,
		0x64, 0x54, 0x54, 0x4f, 0x5a, 0x54, 0x74, 0x47, 0x50, 0x2b,
		0x6b, 0x77, 0x39, 0x75, 0x75, 0x6f, 0x34, 0x39, 0x77, 0x44,
		0x38, 0x34, 0x0a, 0x56, 0x47, 0x4e, 0x4f, 0x62, 0x6d, 0x50,
		0x34, 0x53, 0x31, 0x6c, 0x68, 0x50, 0x61, 0x56, 0x37, 0x77,
		0x79, 0x67, 0x4a, 0x4e, 0x41, 0x75, 0x51, 0x35, 0x49, 0x7a,
		0x71, 0x59, 0x39, 0x41, 0x4a, 0x57, 0x43, 0x66, 0x36, 0x36,
		0x4d, 0x69, 0x49, 0x38, 0x69, 0x62, 0x38, 0x62, 0x70, 0x59,
		0x76, 0x4d, 0x62, 0x59, 0x62, 0x6b, 0x4f, 0x32, 0x62, 0x6a,
		0x51, 0x4b, 0x42, 0x67, 0x45, 0x7a, 0x77, 0x0a, 0x33, 0x45,
		0x33, 0x76, 0x56, 0x70, 0x48, 0x53, 0x6c, 0x55, 0x31, 0x77,
		0x37, 0x50, 0x54, 0x2b, 0x6b, 0x68, 0x70, 0x42, 0x6f, 0x56,
		0x64, 0x79, 0x69, 0x7a, 0x6e, 0x77, 0x50, 0x49, 0x59, 0x4e,
		0x34, 0x42, 0x68, 0x65, 0x37, 0x47, 0x54, 0x74, 0x52, 0x39,
		0x4e, 0x67, 0x4e, 0x42, 0x54, 0x41, 0x72, 0x45, 0x31, 0x4d,
		0x33, 0x74, 0x41, 0x30, 0x6c, 0x79, 0x49, 0x6e, 0x4d, 0x52,
		0x70, 0x75, 0x0a, 0x68, 0x48, 0x53, 0x34, 0x57, 0x75, 0x2b,
		0x47, 0x62, 0x57, 0x44, 0x76, 0x7a, 0x77, 0x67, 0x78, 0x51,
		0x31, 0x31, 0x68, 0x68, 0x71, 0x56, 0x6f, 0x45, 0x44, 0x77,
		0x76, 0x68, 0x76, 0x4e, 0x33, 0x45, 0x67, 0x4a, 0x39, 0x6e,
		0x4a, 0x49, 0x72, 0x6b, 0x79, 0x72, 0x33, 0x53, 0x70, 0x6c,
		0x7a, 0x76, 0x71, 0x4a, 0x70, 0x45, 0x6c, 0x6d, 0x49, 0x68,
		0x70, 0x57, 0x78, 0x52, 0x6a, 0x2f, 0x34, 0x0a, 0x6e, 0x38,
		0x6e, 0x69, 0x47, 0x71, 0x48, 0x31, 0x2f, 0x78, 0x75, 0x30,
		0x31, 0x52, 0x42, 0x4c, 0x32, 0x69, 0x76, 0x2f, 0x6f, 0x71,
		0x53, 0x2b, 0x39, 0x31, 0x59, 0x31, 0x47, 0x70, 0x6a, 0x49,
		0x5a, 0x71, 0x6b, 0x48, 0x62, 0x50, 0x57, 0x5a, 0x41, 0x6f,
		0x47, 0x42, 0x41, 0x49, 0x61, 0x4f, 0x56, 0x6c, 0x50, 0x4b,
		0x41, 0x77, 0x79, 0x6f, 0x2b, 0x67, 0x6d, 0x2f, 0x51, 0x58,
		0x68, 0x75, 0x0a, 0x64, 0x2b, 0x2b, 0x4b, 0x77, 0x38, 0x55,
		0x73, 0x35, 0x74, 0x59, 0x33, 0x54, 0x53, 0x76, 0x43, 0x76,
		0x79, 0x46, 0x38, 0x48, 0x74, 0x37, 0x6e, 0x73, 0x6b, 0x77,
		0x78, 0x58, 0x46, 0x68, 0x78, 0x71, 0x54, 0x42, 0x74, 0x45,
		0x32, 0x55, 0x45, 0x54, 0x6e, 0x76, 0x73, 0x31, 0x76, 0x6f,
		0x33, 0x39, 0x46, 0x44, 0x53, 0x53, 0x57, 0x36, 0x30, 0x4a,
		0x6e, 0x6d, 0x51, 0x32, 0x66, 0x56, 0x4f, 0x0a, 0x50, 0x4a,
		0x37, 0x7a, 0x2f, 0x41, 0x78, 0x6d, 0x52, 0x30, 0x5a, 0x4f,
		0x4b, 0x4b, 0x39, 0x69, 0x4e, 0x4a, 0x68, 0x66, 0x56, 0x68,
		0x49, 0x6d, 0x68, 0x76, 0x52, 0x57, 0x37, 0x4d, 0x2f, 0x53,
		0x33, 0x2b, 0x48, 0x6e, 0x66, 0x4f, 0x32, 0x4d, 0x6b, 0x65,
		0x51, 0x6e, 0x75, 0x7a, 0x4d, 0x2f, 0x53, 0x34, 0x45, 0x43,
		0x52, 0x76, 0x39, 0x64, 0x30, 0x70, 0x55, 0x4a, 0x51, 0x5a,
		0x4b, 0x43, 0x0a, 0x50, 0x58, 0x6f, 0x37, 0x6e, 0x6c, 0x45,
		0x54, 0x6d, 0x64, 0x38, 0x74, 0x66, 0x41, 0x30, 0x6c, 0x74,
		0x59, 0x5a, 0x45, 0x6a, 0x33, 0x57, 0x57, 0x0a, 0x2d, 0x2d,
		0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x50, 0x52, 0x49,
		0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d,
		0x2d, 0x2d, 0x2d, 0x0a,
	}},
};
/*

RSASHA256 (4096b)

rsa4096-256.	DNSKEY	  256  3  8  AwEAAblR1oDSmL2bZLRjaAR9JNetsqhqYlUxzMNQSH2o/CY7WHFZPYqEOBXDBcqLZG+4FPIpm9lZuINjSMkqmCpRTNoL2f3KGE5tj+5Cqq5XN24BDMgsZ5foMHMDaqjewBr1wSU5M/End28iogvA5wgRw6OgbkmIyLg5UCxc9FpO07UW6bb9O7DUwZjbRg8EjUGVxuL8eH8Az6sKinxVEMYeHfV8BhidQ0ag4cGmu0fymn2JQzJY/Y/aLNGvHZ1e8fcK21eJ5hCSBg5ds0XK9yuvJowloyMhihSaPQUMO2vxn7jEBZaUCkL9oWu0KFSl6KCYmyxZQzwyg2bwzL/dF2/Xgbh9xFF4tGiNw97vEfP+ha3kKXZNH9wfKrZWavpQL1flrOqjBC4THFvLJQAbx9446MUGxUwPh4Pia//H0NYBN8FPNxcOGg+73VVSkfYH3pSIYijtss93dQoSZcO/92FL0U3HjcwtjVvOhDScPy1qkbGkgBFLuz5aZLqDzVF6g8AG1VePNv3o31UqomI5pUtATY0T62uw1oZuB+6NcQwsrO6Dd8L+FSGmtGC93xJqpXrtG/UMLvsYWweIPIk0J8zaoEE1TaA2M2WTbOWReJyl7Ps//gpILlkvzwNnaADlVOBsL9CsEy3npJeqzI9IBDlsj6gmWrgFdgvJDITM7y1Xouwh
rsa4096-256.	DS    	34375  8  1  D06091F84E5BCA92760D43CB07AD50CCFABAAE40
rsa4096-256.	DS    	34375  8  2  505CD3A992BC0DE274F45A012C40BF094217969C5D5BDE2E46E9AC0028326F52
rsa4096-256.	DS    	34375  8  4  DCF941EE81BF9AC8EF75FABE6732E9C2BEE56DC9E50C1E5E11F6C4B36AF2206811ED3DEA99DAD267E8F261A17A21DE3A

-----BEGIN PRIVATE KEY-----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-----END PRIVATE KEY-----

*/
static const key_parameters_t SAMPLE_RSA4096_SHA256_KEY = {
	.name = (uint8_t *)"\x0b""rsa4096-256",
	.flags = 256,
	.protocol = 3,
	.algorithm = 8,
	.public_key = { .size = 516, .data = (uint8_t []){
		0x03, 0x01, 0x00, 0x01, 0xb9, 0x51, 0xd6, 0x80, 0xd2, 0x98,
		0xbd, 0x9b, 0x64, 0xb4, 0x63, 0x68, 0x04, 0x7d, 0x24, 0xd7,
		0xad, 0xb2, 0xa8, 0x6a, 0x62, 0x55, 0x31, 0xcc, 0xc3, 0x50,
		0x48, 0x7d, 0xa8, 0xfc, 0x26, 0x3b, 0x58, 0x71, 0x59, 0x3d,
		0x8a, 0x84, 0x38, 0x15, 0xc3, 0x05, 0xca, 0x8b, 0x64, 0x6f,
		0xb8, 0x14, 0xf2, 0x29, 0x9b, 0xd9, 0x59, 0xb8, 0x83, 0x63,
		0x48, 0xc9, 0x2a, 0x98, 0x2a, 0x51, 0x4c, 0xda, 0x0b, 0xd9,
		0xfd, 0xca, 0x18, 0x4e, 0x6d, 0x8f, 0xee, 0x42, 0xaa, 0xae,
		0x57, 0x37, 0x6e, 0x01, 0x0c, 0xc8, 0x2c, 0x67, 0x97, 0xe8,
		0x30, 0x73, 0x03, 0x6a, 0xa8, 0xde, 0xc0, 0x1a, 0xf5, 0xc1,
		0x25, 0x39, 0x33, 0xf1, 0x27, 0x77, 0x6f, 0x22, 0xa2, 0x0b,
		0xc0, 0xe7, 0x08, 0x11, 0xc3, 0xa3, 0xa0, 0x6e, 0x49, 0x88,
		0xc8, 0xb8, 0x39, 0x50, 0x2c, 0x5c, 0xf4, 0x5a, 0x4e, 0xd3,
		0xb5, 0x16, 0xe9, 0xb6, 0xfd, 0x3b, 0xb0, 0xd4, 0xc1, 0x98,
		0xdb, 0x46, 0x0f, 0x04, 0x8d, 0x41, 0x95, 0xc6, 0xe2, 0xfc,
		0x78, 0x7f, 0x00, 0xcf, 0xab, 0x0a, 0x8a, 0x7c, 0x55, 0x10,
		0xc6, 0x1e, 0x1d, 0xf5, 0x7c, 0x06, 0x18, 0x9d, 0x43, 0x46,
		0xa0, 0xe1, 0xc1, 0xa6, 0xbb, 0x47, 0xf2, 0x9a, 0x7d, 0x89,
		0x43, 0x32, 0x58, 0xfd, 0x8f, 0xda, 0x2c, 0xd1, 0xaf, 0x1d,
		0x9d, 0x5e, 0xf1, 0xf7, 0x0a, 0xdb, 0x57, 0x89, 0xe6, 0x10,
		0x92, 0x06, 0x0e, 0x5d, 0xb3, 0x45, 0xca, 0xf7, 0x2b, 0xaf,
		0x26, 0x8c, 0x25, 0xa3, 0x23, 0x21, 0x8a, 0x14, 0x9a, 0x3d,
		0x05, 0x0c, 0x3b, 0x6b, 0xf1, 0x9f, 0xb8, 0xc4, 0x05, 0x96,
		0x94, 0x0a, 0x42, 0xfd, 0xa1, 0x6b, 0xb4, 0x28, 0x54, 0xa5,
		0xe8, 0xa0, 0x98, 0x9b, 0x2c, 0x59, 0x43, 0x3c, 0x32, 0x83,
		0x66, 0xf0, 0xcc, 0xbf, 0xdd, 0x17, 0x6f, 0xd7, 0x81, 0xb8,
		0x7d, 0xc4, 0x51, 0x78, 0xb4, 0x68, 0x8d, 0xc3, 0xde, 0xef,
		0x11, 0xf3, 0xfe, 0x85, 0xad, 0xe4, 0x29, 0x76, 0x4d, 0x1f,
		0xdc, 0x1f, 0x2a, 0xb6, 0x56, 0x6a, 0xfa, 0x50, 0x2f, 0x57,
		0xe5, 0xac, 0xea, 0xa3, 0x04, 0x2e, 0x13, 0x1c, 0x5b, 0xcb,
		0x25, 0x00, 0x1b, 0xc7, 0xde, 0x38, 0xe8, 0xc5, 0x06, 0xc5,
		0x4c, 0x0f, 0x87, 0x83, 0xe2, 0x6b, 0xff, 0xc7, 0xd0, 0xd6,
		0x01, 0x37, 0xc1, 0x4f, 0x37, 0x17, 0x0e, 0x1a, 0x0f, 0xbb,
		0xdd, 0x55, 0x52, 0x91, 0xf6, 0x07, 0xde, 0x94, 0x88, 0x62,
		0x28, 0xed, 0xb2, 0xcf, 0x77, 0x75, 0x0a, 0x12, 0x65, 0xc3,
		0xbf, 0xf7, 0x61, 0x4b, 0xd1, 0x4d, 0xc7, 0x8d, 0xcc, 0x2d,
		0x8d, 0x5b, 0xce, 0x84, 0x34, 0x9c, 0x3f, 0x2d, 0x6a, 0x91,
		0xb1, 0xa4, 0x80, 0x11, 0x4b, 0xbb, 0x3e, 0x5a, 0x64, 0xba,
		0x83, 0xcd, 0x51, 0x7a, 0x83, 0xc0, 0x06, 0xd5, 0x57, 0x8f,
		0x36, 0xfd, 0xe8, 0xdf, 0x55, 0x2a, 0xa2, 0x62, 0x39, 0xa5,
		0x4b, 0x40, 0x4d, 0x8d, 0x13, 0xeb, 0x6b, 0xb0, 0xd6, 0x86,
		0x6e, 0x07, 0xee, 0x8d, 0x71, 0x0c, 0x2c, 0xac, 0xee, 0x83,
		0x77, 0xc2, 0xfe, 0x15, 0x21, 0xa6, 0xb4, 0x60, 0xbd, 0xdf,
		0x12, 0x6a, 0xa5, 0x7a, 0xed, 0x1b, 0xf5, 0x0c, 0x2e, 0xfb,
		0x18, 0x5b, 0x07, 0x88, 0x3c, 0x89, 0x34, 0x27, 0xcc, 0xda,
		0xa0, 0x41, 0x35, 0x4d, 0xa0, 0x36, 0x33, 0x65, 0x93, 0x6c,
		0xe5, 0x91, 0x78, 0x9c, 0xa5, 0xec, 0xfb, 0x3f, 0xfe, 0x0a,
		0x48, 0x2e, 0x59, 0x2f, 0xcf, 0x03, 0x67, 0x68, 0x00, 0xe5,
		0x54, 0xe0, 0x6c, 0x2f, 0xd0, 0xac, 0x13, 0x2d, 0xe7, 0xa4,
		0x97, 0xaa, 0xcc, 0x8f, 0x48, 0x04, 0x39, 0x6c, 0x8f, 0xa8,
		0x26, 0x5a, 0xb8, 0x05, 0x76, 0x0b, 0xc9, 0x0c, 0x84, 0xcc,
		0xef, 0x2d, 0x57, 0xa2, 0xec, 0x21,
	}},
	.rdata = { .size = 520, .data = (uint8_t []){
		01, 0x00, 0x03, 0x08,
		0x03, 0x01, 0x00, 0x01, 0xb9, 0x51, 0xd6, 0x80, 0xd2, 0x98,
		0xbd, 0x9b, 0x64, 0xb4, 0x63, 0x68, 0x04, 0x7d, 0x24, 0xd7,
		0xad, 0xb2, 0xa8, 0x6a, 0x62, 0x55, 0x31, 0xcc, 0xc3, 0x50,
		0x48, 0x7d, 0xa8, 0xfc, 0x26, 0x3b, 0x58, 0x71, 0x59, 0x3d,
		0x8a, 0x84, 0x38, 0x15, 0xc3, 0x05, 0xca, 0x8b, 0x64, 0x6f,
		0xb8, 0x14, 0xf2, 0x29, 0x9b, 0xd9, 0x59, 0xb8, 0x83, 0x63,
		0x48, 0xc9, 0x2a, 0x98, 0x2a, 0x51, 0x4c, 0xda, 0x0b, 0xd9,
		0xfd, 0xca, 0x18, 0x4e, 0x6d, 0x8f, 0xee, 0x42, 0xaa, 0xae,
		0x57, 0x37, 0x6e, 0x01, 0x0c, 0xc8, 0x2c, 0x67, 0x97, 0xe8,
		0x30, 0x73, 0x03, 0x6a, 0xa8, 0xde, 0xc0, 0x1a, 0xf5, 0xc1,
		0x25, 0x39, 0x33, 0xf1, 0x27, 0x77, 0x6f, 0x22, 0xa2, 0x0b,
		0xc0, 0xe7, 0x08, 0x11, 0xc3, 0xa3, 0xa0, 0x6e, 0x49, 0x88,
		0xc8, 0xb8, 0x39, 0x50, 0x2c, 0x5c, 0xf4, 0x5a, 0x4e, 0xd3,
		0xb5, 0x16, 0xe9, 0xb6, 0xfd, 0x3b, 0xb0, 0xd4, 0xc1, 0x98,
		0xdb, 0x46, 0x0f, 0x04, 0x8d, 0x41, 0x95, 0xc6, 0xe2, 0xfc,
		0x78, 0x7f, 0x00, 0xcf, 0xab, 0x0a, 0x8a, 0x7c, 0x55, 0x10,
		0xc6, 0x1e, 0x1d, 0xf5, 0x7c, 0x06, 0x18, 0x9d, 0x43, 0x46,
		0xa0, 0xe1, 0xc1, 0xa6, 0xbb, 0x47, 0xf2, 0x9a, 0x7d, 0x89,
		0x43, 0x32, 0x58, 0xfd, 0x8f, 0xda, 0x2c, 0xd1, 0xaf, 0x1d,
		0x9d, 0x5e, 0xf1, 0xf7, 0x0a, 0xdb, 0x57, 0x89, 0xe6, 0x10,
		0x92, 0x06, 0x0e, 0x5d, 0xb3, 0x45, 0xca, 0xf7, 0x2b, 0xaf,
		0x26, 0x8c, 0x25, 0xa3, 0x23, 0x21, 0x8a, 0x14, 0x9a, 0x3d,
		0x05, 0x0c, 0x3b, 0x6b, 0xf1, 0x9f, 0xb8, 0xc4, 0x05, 0x96,
		0x94, 0x0a, 0x42, 0xfd, 0xa1, 0x6b, 0xb4, 0x28, 0x54, 0xa5,
		0xe8, 0xa0, 0x98, 0x9b, 0x2c, 0x59, 0x43, 0x3c, 0x32, 0x83,
		0x66, 0xf0, 0xcc, 0xbf, 0xdd, 0x17, 0x6f, 0xd7, 0x81, 0xb8,
		0x7d, 0xc4, 0x51, 0x78, 0xb4, 0x68, 0x8d, 0xc3, 0xde, 0xef,
		0x11, 0xf3, 0xfe, 0x85, 0xad, 0xe4, 0x29, 0x76, 0x4d, 0x1f,
		0xdc, 0x1f, 0x2a, 0xb6, 0x56, 0x6a, 0xfa, 0x50, 0x2f, 0x57,
		0xe5, 0xac, 0xea, 0xa3, 0x04, 0x2e, 0x13, 0x1c, 0x5b, 0xcb,
		0x25, 0x00, 0x1b, 0xc7, 0xde, 0x38, 0xe8, 0xc5, 0x06, 0xc5,
		0x4c, 0x0f, 0x87, 0x83, 0xe2, 0x6b, 0xff, 0xc7, 0xd0, 0xd6,
		0x01, 0x37, 0xc1, 0x4f, 0x37, 0x17, 0x0e, 0x1a, 0x0f, 0xbb,
		0xdd, 0x55, 0x52, 0x91, 0xf6, 0x07, 0xde, 0x94, 0x88, 0x62,
		0x28, 0xed, 0xb2, 0xcf, 0x77, 0x75, 0x0a, 0x12, 0x65, 0xc3,
		0xbf, 0xf7, 0x61, 0x4b, 0xd1, 0x4d, 0xc7, 0x8d, 0xcc, 0x2d,
		0x8d, 0x5b, 0xce, 0x84, 0x34, 0x9c, 0x3f, 0x2d, 0x6a, 0x91,
		0xb1, 0xa4, 0x80, 0x11, 0x4b, 0xbb, 0x3e, 0x5a, 0x64, 0xba,
		0x83, 0xcd, 0x51, 0x7a, 0x83, 0xc0, 0x06, 0xd5, 0x57, 0x8f,
		0x36, 0xfd, 0xe8, 0xdf, 0x55, 0x2a, 0xa2, 0x62, 0x39, 0xa5,
		0x4b, 0x40, 0x4d, 0x8d, 0x13, 0xeb, 0x6b, 0xb0, 0xd6, 0x86,
		0x6e, 0x07, 0xee, 0x8d, 0x71, 0x0c, 0x2c, 0xac, 0xee, 0x83,
		0x77, 0xc2, 0xfe, 0x15, 0x21, 0xa6, 0xb4, 0x60, 0xbd, 0xdf,
		0x12, 0x6a, 0xa5, 0x7a, 0xed, 0x1b, 0xf5, 0x0c, 0x2e, 0xfb,
		0x18, 0x5b, 0x07, 0x88, 0x3c, 0x89, 0x34, 0x27, 0xcc, 0xda,
		0xa0, 0x41, 0x35, 0x4d, 0xa0, 0x36, 0x33, 0x65, 0x93, 0x6c,
		0xe5, 0x91, 0x78, 0x9c, 0xa5, 0xec, 0xfb, 0x3f, 0xfe, 0x0a,
		0x48, 0x2e, 0x59, 0x2f, 0xcf, 0x03, 0x67, 0x68, 0x00, 0xe5,
		0x54, 0xe0, 0x6c, 0x2f, 0xd0, 0xac, 0x13, 0x2d, 0xe7, 0xa4,
		0x97, 0xaa, 0xcc, 0x8f, 0x48, 0x04, 0x39, 0x6c, 0x8f, 0xa8,
		0x26, 0x5a, 0xb8, 0x05, 0x76, 0x0b, 0xc9, 0x0c, 0x84, 0xcc,
		0xef, 0x2d, 0x57, 0xa2, 0xec, 0x21,
	}},
	.key_id = "07aa99ffce866f7b37bfcc43406d621bfb6f707a",
	.keytag = 34375,
	.ds_sha1 = { .size = 24, .data = (uint8_t []){
		0x86, 0x47, 0x08, 0x01, 0xd0, 0x60, 0x91, 0xf8, 0x4e, 0x5b,
		0xca, 0x92, 0x76, 0x0d, 0x43, 0xcb, 0x07, 0xad, 0x50, 0xcc,
		0xfa, 0xba, 0xae, 0x40,
	}},
	.ds_sha256 = { .size = 36, .data = (uint8_t []){
		0x86, 0x47, 0x08, 0x02, 0x50, 0x5c, 0xd3, 0xa9, 0x92, 0xbc,
		0x0d, 0xe2, 0x74, 0xf4, 0x5a, 0x01, 0x2c, 0x40, 0xbf, 0x09,
		0x42, 0x17, 0x96, 0x9c, 0x5d, 0x5b, 0xde, 0x2e, 0x46, 0xe9,
		0xac, 0x00, 0x28, 0x32, 0x6f, 0x52,
	}},
	.ds_sha384 = { .size = 52, .data = (uint8_t []){
		0x86, 0x47, 0x08, 0x04, 0xdc, 0xf9, 0x41, 0xee, 0x81, 0xbf,
		0x9a, 0xc8, 0xef, 0x75, 0xfa, 0xbe, 0x67, 0x32, 0xe9, 0xc2,
		0xbe, 0xe5, 0x6d, 0xc9, 0xe5, 0x0c, 0x1e, 0x5e, 0x11, 0xf6,
		0xc4, 0xb3, 0x6a, 0xf2, 0x20, 0x68, 0x11, 0xed, 0x3d, 0xea,
		0x99, 0xda, 0xd2, 0x67, 0xe8, 0xf2, 0x61, 0xa1, 0x7a, 0x21,
		0xde, 0x3a,
	}},
	.bit_size = 4096,
	.pem = { .size = 3272, .data = (uint8_t []){
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49,
		0x49, 0x4a, 0x51, 0x67, 0x49, 0x42, 0x41, 0x44, 0x41, 0x4e,
		0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77,
		0x30, 0x42, 0x41, 0x51, 0x45, 0x46, 0x41, 0x41, 0x53, 0x43,
		0x43, 0x53, 0x77, 0x77, 0x67, 0x67, 0x6b, 0x6f, 0x41, 0x67,
		0x45, 0x41, 0x41, 0x6f, 0x49, 0x43, 0x41, 0x51, 0x43, 0x35,
		0x55, 0x64, 0x61, 0x41, 0x30, 0x70, 0x69, 0x39, 0x6d, 0x32,
		0x53, 0x30, 0x0a, 0x59, 0x32, 0x67, 0x45, 0x66, 0x53, 0x54,
		0x58, 0x72, 0x62, 0x4b, 0x6f, 0x61, 0x6d, 0x4a, 0x56, 0x4d,
		0x63, 0x7a, 0x44, 0x55, 0x45, 0x68, 0x39, 0x71, 0x50, 0x77,
		0x6d, 0x4f, 0x31, 0x68, 0x78, 0x57, 0x54, 0x32, 0x4b, 0x68,
		0x44, 0x67, 0x56, 0x77, 0x77, 0x58, 0x4b, 0x69, 0x32, 0x52,
		0x76, 0x75, 0x42, 0x54, 0x79, 0x4b, 0x5a, 0x76, 0x5a, 0x57,
		0x62, 0x69, 0x44, 0x59, 0x30, 0x6a, 0x4a, 0x0a, 0x4b, 0x70,
		0x67, 0x71, 0x55, 0x55, 0x7a, 0x61, 0x43, 0x39, 0x6e, 0x39,
		0x79, 0x68, 0x68, 0x4f, 0x62, 0x59, 0x2f, 0x75, 0x51, 0x71,
		0x71, 0x75, 0x56, 0x7a, 0x64, 0x75, 0x41, 0x51, 0x7a, 0x49,
		0x4c, 0x47, 0x65, 0x58, 0x36, 0x44, 0x42, 0x7a, 0x41, 0x32,
		0x71, 0x6f, 0x33, 0x73, 0x41, 0x61, 0x39, 0x63, 0x45, 0x6c,
		0x4f, 0x54, 0x50, 0x78, 0x4a, 0x33, 0x64, 0x76, 0x49, 0x71,
		0x49, 0x4c, 0x0a, 0x77, 0x4f, 0x63, 0x49, 0x45, 0x63, 0x4f,
		0x6a, 0x6f, 0x47, 0x35, 0x4a, 0x69, 0x4d, 0x69, 0x34, 0x4f,
		0x56, 0x41, 0x73, 0x58, 0x50, 0x52, 0x61, 0x54, 0x74, 0x4f,
		0x31, 0x46, 0x75, 0x6d, 0x32, 0x2f, 0x54, 0x75, 0x77, 0x31,
		0x4d, 0x47, 0x59, 0x32, 0x30, 0x59, 0x50, 0x42, 0x49, 0x31,
		0x42, 0x6c, 0x63, 0x62, 0x69, 0x2f, 0x48, 0x68, 0x2f, 0x41,
		0x4d, 0x2b, 0x72, 0x43, 0x6f, 0x70, 0x38, 0x0a, 0x56, 0x52,
		0x44, 0x47, 0x48, 0x68, 0x33, 0x31, 0x66, 0x41, 0x59, 0x59,
		0x6e, 0x55, 0x4e, 0x47, 0x6f, 0x4f, 0x48, 0x42, 0x70, 0x72,
		0x74, 0x48, 0x38, 0x70, 0x70, 0x39, 0x69, 0x55, 0x4d, 0x79,
		0x57, 0x50, 0x32, 0x50, 0x32, 0x69, 0x7a, 0x52, 0x72, 0x78,
		0x32, 0x64, 0x58, 0x76, 0x48, 0x33, 0x43, 0x74, 0x74, 0x58,
		0x69, 0x65, 0x59, 0x51, 0x6b, 0x67, 0x59, 0x4f, 0x58, 0x62,
		0x4e, 0x46, 0x0a, 0x79, 0x76, 0x63, 0x72, 0x72, 0x79, 0x61,
		0x4d, 0x4a, 0x61, 0x4d, 0x6a, 0x49, 0x59, 0x6f, 0x55, 0x6d,
		0x6a, 0x30, 0x46, 0x44, 0x44, 0x74, 0x72, 0x38, 0x5a, 0x2b,
		0x34, 0x78, 0x41, 0x57, 0x57, 0x6c, 0x41, 0x70, 0x43, 0x2f,
		0x61, 0x46, 0x72, 0x74, 0x43, 0x68, 0x55, 0x70, 0x65, 0x69,
		0x67, 0x6d, 0x4a, 0x73, 0x73, 0x57, 0x55, 0x4d, 0x38, 0x4d,
		0x6f, 0x4e, 0x6d, 0x38, 0x4d, 0x79, 0x2f, 0x0a, 0x33, 0x52,
		0x64, 0x76, 0x31, 0x34, 0x47, 0x34, 0x66, 0x63, 0x52, 0x52,
		0x65, 0x4c, 0x52, 0x6f, 0x6a, 0x63, 0x50, 0x65, 0x37, 0x78,
		0x48, 0x7a, 0x2f, 0x6f, 0x57, 0x74, 0x35, 0x43, 0x6c, 0x32,
		0x54, 0x52, 0x2f, 0x63, 0x48, 0x79, 0x71, 0x32, 0x56, 0x6d,
		0x72, 0x36, 0x55, 0x43, 0x39, 0x58, 0x35, 0x61, 0x7a, 0x71,
		0x6f, 0x77, 0x51, 0x75, 0x45, 0x78, 0x78, 0x62, 0x79, 0x79,
		0x55, 0x41, 0x0a, 0x47, 0x38, 0x66, 0x65, 0x4f, 0x4f, 0x6a,
		0x46, 0x42, 0x73, 0x56, 0x4d, 0x44, 0x34, 0x65, 0x44, 0x34,
		0x6d, 0x76, 0x2f, 0x78, 0x39, 0x44, 0x57, 0x41, 0x54, 0x66,
		0x42, 0x54, 0x7a, 0x63, 0x58, 0x44, 0x68, 0x6f, 0x50, 0x75,
		0x39, 0x31, 0x56, 0x55, 0x70, 0x48, 0x32, 0x42, 0x39, 0x36,
		0x55, 0x69, 0x47, 0x49, 0x6f, 0x37, 0x62, 0x4c, 0x50, 0x64,
		0x33, 0x55, 0x4b, 0x45, 0x6d, 0x58, 0x44, 0x0a, 0x76, 0x2f,
		0x64, 0x68, 0x53, 0x39, 0x46, 0x4e, 0x78, 0x34, 0x33, 0x4d,
		0x4c, 0x59, 0x31, 0x62, 0x7a, 0x6f, 0x51, 0x30, 0x6e, 0x44,
		0x38, 0x74, 0x61, 0x70, 0x47, 0x78, 0x70, 0x49, 0x41, 0x52,
		0x53, 0x37, 0x73, 0x2b, 0x57, 0x6d, 0x53, 0x36, 0x67, 0x38,
		0x31, 0x52, 0x65, 0x6f, 0x50, 0x41, 0x42, 0x74, 0x56, 0x58,
		0x6a, 0x7a, 0x62, 0x39, 0x36, 0x4e, 0x39, 0x56, 0x4b, 0x71,
		0x4a, 0x69, 0x0a, 0x4f, 0x61, 0x56, 0x4c, 0x51, 0x45, 0x32,
		0x4e, 0x45, 0x2b, 0x74, 0x72, 0x73, 0x4e, 0x61, 0x47, 0x62,
		0x67, 0x66, 0x75, 0x6a, 0x58, 0x45, 0x4d, 0x4c, 0x4b, 0x7a,
		0x75, 0x67, 0x33, 0x66, 0x43, 0x2f, 0x68, 0x55, 0x68, 0x70,
		0x72, 0x52, 0x67, 0x76, 0x64, 0x38, 0x53, 0x61, 0x71, 0x56,
		0x36, 0x37, 0x52, 0x76, 0x31, 0x44, 0x43, 0x37, 0x37, 0x47,
		0x46, 0x73, 0x48, 0x69, 0x44, 0x79, 0x4a, 0x0a, 0x4e, 0x43,
		0x66, 0x4d, 0x32, 0x71, 0x42, 0x42, 0x4e, 0x55, 0x32, 0x67,
		0x4e, 0x6a, 0x4e, 0x6c, 0x6b, 0x32, 0x7a, 0x6c, 0x6b, 0x58,
		0x69, 0x63, 0x70, 0x65, 0x7a, 0x37, 0x50, 0x2f, 0x34, 0x4b,
		0x53, 0x43, 0x35, 0x5a, 0x4c, 0x38, 0x38, 0x44, 0x5a, 0x32,
		0x67, 0x41, 0x35, 0x56, 0x54, 0x67, 0x62, 0x43, 0x2f, 0x51,
		0x72, 0x42, 0x4d, 0x74, 0x35, 0x36, 0x53, 0x58, 0x71, 0x73,
		0x79, 0x50, 0x0a, 0x53, 0x41, 0x51, 0x35, 0x62, 0x49, 0x2b,
		0x6f, 0x4a, 0x6c, 0x71, 0x34, 0x42, 0x58, 0x59, 0x4c, 0x79,
		0x51, 0x79, 0x45, 0x7a, 0x4f, 0x38, 0x74, 0x56, 0x36, 0x4c,
		0x73, 0x49, 0x51, 0x49, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41,
		0x6f, 0x49, 0x42, 0x2f, 0x33, 0x4d, 0x5a, 0x59, 0x53, 0x43,
		0x68, 0x72, 0x44, 0x77, 0x58, 0x52, 0x33, 0x54, 0x43, 0x53,
		0x4e, 0x69, 0x41, 0x34, 0x57, 0x45, 0x54, 0x0a, 0x67, 0x4f,
		0x4d, 0x5a, 0x4c, 0x2f, 0x6d, 0x35, 0x39, 0x48, 0x4e, 0x61,
		0x61, 0x61, 0x75, 0x69, 0x53, 0x50, 0x6c, 0x31, 0x34, 0x6e,
		0x69, 0x44, 0x39, 0x4c, 0x55, 0x54, 0x61, 0x63, 0x49, 0x61,
		0x58, 0x6e, 0x35, 0x75, 0x46, 0x50, 0x4e, 0x4f, 0x59, 0x62,
		0x51, 0x47, 0x49, 0x38, 0x58, 0x61, 0x38, 0x37, 0x36, 0x6d,
		0x36, 0x34, 0x34, 0x46, 0x35, 0x6e, 0x78, 0x69, 0x41, 0x6c,
		0x6c, 0x72, 0x0a, 0x41, 0x76, 0x6d, 0x64, 0x69, 0x33, 0x73,
		0x71, 0x6b, 0x54, 0x70, 0x71, 0x4c, 0x5a, 0x67, 0x61, 0x6b,
		0x31, 0x4b, 0x2f, 0x56, 0x44, 0x44, 0x2f, 0x59, 0x31, 0x66,
		0x31, 0x33, 0x69, 0x47, 0x51, 0x2f, 0x61, 0x33, 0x6b, 0x71,
		0x32, 0x77, 0x54, 0x4e, 0x75, 0x4c, 0x6e, 0x70, 0x46, 0x73,
		0x6a, 0x4b, 0x41, 0x72, 0x65, 0x64, 0x38, 0x62, 0x56, 0x43,
		0x4b, 0x6a, 0x36, 0x58, 0x34, 0x68, 0x42, 0x0a, 0x48, 0x44,
		0x36, 0x50, 0x74, 0x68, 0x71, 0x31, 0x66, 0x68, 0x59, 0x50,
		0x70, 0x62, 0x52, 0x78, 0x51, 0x51, 0x77, 0x34, 0x4c, 0x76,
		0x42, 0x34, 0x56, 0x35, 0x70, 0x38, 0x69, 0x63, 0x46, 0x6d,
		0x62, 0x78, 0x7a, 0x6b, 0x74, 0x74, 0x63, 0x46, 0x4c, 0x34,
		0x48, 0x4c, 0x54, 0x51, 0x75, 0x55, 0x64, 0x57, 0x62, 0x47,
		0x4a, 0x54, 0x30, 0x54, 0x6d, 0x46, 0x64, 0x30, 0x58, 0x53,
		0x63, 0x61, 0x0a, 0x34, 0x4b, 0x35, 0x6f, 0x2f, 0x6c, 0x2f,
		0x66, 0x6a, 0x46, 0x65, 0x36, 0x4e, 0x70, 0x64, 0x35, 0x46,
		0x6e, 0x72, 0x68, 0x78, 0x64, 0x70, 0x2f, 0x61, 0x76, 0x7a,
		0x74, 0x68, 0x4f, 0x69, 0x32, 0x32, 0x71, 0x33, 0x6a, 0x53,
		0x6d, 0x55, 0x6c, 0x57, 0x52, 0x59, 0x46, 0x6a, 0x39, 0x36,
		0x36, 0x30, 0x77, 0x67, 0x5a, 0x4b, 0x6a, 0x38, 0x75, 0x30,
		0x46, 0x56, 0x52, 0x63, 0x35, 0x2b, 0x58, 0x0a, 0x77, 0x4c,
		0x73, 0x52, 0x47, 0x62, 0x47, 0x79, 0x47, 0x6a, 0x51, 0x49,
		0x5a, 0x4e, 0x2b, 0x6d, 0x4d, 0x6f, 0x32, 0x50, 0x4d, 0x65,
		0x6c, 0x55, 0x43, 0x51, 0x75, 0x53, 0x65, 0x36, 0x39, 0x70,
		0x4e, 0x61, 0x4e, 0x64, 0x61, 0x42, 0x35, 0x57, 0x71, 0x67,
		0x6e, 0x68, 0x5a, 0x75, 0x44, 0x38, 0x66, 0x71, 0x42, 0x77,
		0x37, 0x72, 0x6f, 0x37, 0x6c, 0x31, 0x4c, 0x66, 0x68, 0x72,
		0x45, 0x41, 0x0a, 0x61, 0x2f, 0x71, 0x62, 0x48, 0x36, 0x61,
		0x56, 0x38, 0x38, 0x47, 0x52, 0x36, 0x4e, 0x2b, 0x72, 0x4c,
		0x51, 0x4c, 0x62, 0x4b, 0x35, 0x43, 0x30, 0x75, 0x49, 0x31,
		0x41, 0x35, 0x33, 0x39, 0x32, 0x43, 0x6e, 0x36, 0x5a, 0x4b,
		0x32, 0x62, 0x69, 0x50, 0x79, 0x30, 0x56, 0x47, 0x6d, 0x4b,
		0x61, 0x61, 0x71, 0x53, 0x77, 0x68, 0x57, 0x71, 0x44, 0x74,
		0x74, 0x30, 0x43, 0x42, 0x6c, 0x48, 0x36, 0x0a, 0x50, 0x2b,
		0x78, 0x39, 0x6b, 0x2b, 0x4b, 0x36, 0x58, 0x67, 0x73, 0x6e,
		0x47, 0x48, 0x74, 0x6b, 0x4a, 0x4c, 0x79, 0x38, 0x6e, 0x57,
		0x75, 0x66, 0x31, 0x6b, 0x41, 0x54, 0x78, 0x70, 0x46, 0x50,
		0x4a, 0x4f, 0x73, 0x6f, 0x73, 0x47, 0x55, 0x55, 0x30, 0x50,
		0x53, 0x39, 0x54, 0x34, 0x77, 0x57, 0x72, 0x6f, 0x38, 0x47,
		0x56, 0x65, 0x47, 0x5a, 0x30, 0x58, 0x63, 0x55, 0x79, 0x2b,
		0x37, 0x4c, 0x0a, 0x72, 0x49, 0x63, 0x34, 0x6f, 0x47, 0x70,
		0x77, 0x50, 0x6f, 0x6d, 0x4f, 0x63, 0x67, 0x6e, 0x2f, 0x77,
		0x64, 0x30, 0x4a, 0x4a, 0x66, 0x62, 0x6f, 0x52, 0x41, 0x56,
		0x47, 0x4b, 0x6a, 0x78, 0x65, 0x76, 0x37, 0x55, 0x6c, 0x2f,
		0x42, 0x75, 0x52, 0x61, 0x6d, 0x52, 0x68, 0x2b, 0x48, 0x4d,
		0x6a, 0x76, 0x2b, 0x42, 0x65, 0x45, 0x75, 0x69, 0x34, 0x50,
		0x41, 0x65, 0x68, 0x6d, 0x35, 0x2b, 0x6a, 0x0a, 0x32, 0x36,
		0x7a, 0x78, 0x68, 0x32, 0x44, 0x59, 0x2b, 0x57, 0x38, 0x50,
		0x30, 0x59, 0x43, 0x6d, 0x52, 0x79, 0x74, 0x53, 0x64, 0x72,
		0x33, 0x67, 0x4a, 0x31, 0x4c, 0x53, 0x4e, 0x34, 0x72, 0x32,
		0x49, 0x41, 0x4b, 0x43, 0x59, 0x4b, 0x79, 0x54, 0x61, 0x39,
		0x31, 0x63, 0x61, 0x37, 0x53, 0x38, 0x67, 0x46, 0x73, 0x36,
		0x54, 0x6c, 0x49, 0x6e, 0x57, 0x4f, 0x61, 0x52, 0x6d, 0x35,
		0x32, 0x53, 0x0a, 0x4b, 0x2b, 0x6b, 0x43, 0x59, 0x57, 0x43,
		0x58, 0x56, 0x54, 0x6a, 0x50, 0x74, 0x2f, 0x2f, 0x77, 0x73,
		0x4e, 0x62, 0x58, 0x49, 0x33, 0x77, 0x49, 0x66, 0x51, 0x67,
		0x63, 0x4d, 0x76, 0x47, 0x35, 0x74, 0x6f, 0x33, 0x4f, 0x4e,
		0x30, 0x72, 0x2f, 0x31, 0x5a, 0x50, 0x49, 0x6b, 0x56, 0x70,
		0x46, 0x57, 0x79, 0x77, 0x47, 0x39, 0x48, 0x4b, 0x30, 0x78,
		0x57, 0x50, 0x64, 0x69, 0x54, 0x53, 0x4d, 0x0a, 0x54, 0x63,
		0x51, 0x4e, 0x4f, 0x53, 0x51, 0x50, 0x64, 0x42, 0x63, 0x6e,
		0x37, 0x2b, 0x73, 0x70, 0x74, 0x39, 0x45, 0x43, 0x67, 0x67,
		0x45, 0x42, 0x41, 0x4d, 0x33, 0x47, 0x39, 0x4f, 0x4b, 0x36,
		0x6f, 0x52, 0x44, 0x53, 0x2f, 0x54, 0x46, 0x70, 0x30, 0x6c,
		0x33, 0x6d, 0x75, 0x49, 0x54, 0x6a, 0x62, 0x31, 0x6a, 0x4e,
		0x2b, 0x4f, 0x31, 0x73, 0x59, 0x79, 0x67, 0x61, 0x70, 0x4f,
		0x71, 0x64, 0x0a, 0x50, 0x6a, 0x6d, 0x61, 0x64, 0x49, 0x32,
		0x36, 0x77, 0x51, 0x71, 0x7a, 0x48, 0x6b, 0x30, 0x66, 0x65,
		0x4e, 0x41, 0x55, 0x65, 0x54, 0x68, 0x2f, 0x67, 0x54, 0x61,
		0x67, 0x73, 0x57, 0x2f, 0x2b, 0x4a, 0x73, 0x58, 0x49, 0x57,
		0x63, 0x4b, 0x51, 0x4e, 0x34, 0x6e, 0x6f, 0x49, 0x5a, 0x79,
		0x35, 0x2f, 0x50, 0x74, 0x70, 0x78, 0x71, 0x66, 0x63, 0x5a,
		0x66, 0x2f, 0x4f, 0x50, 0x2b, 0x4f, 0x45, 0x0a, 0x71, 0x4c,
		0x78, 0x48, 0x74, 0x4c, 0x51, 0x69, 0x77, 0x52, 0x38, 0x6b,
		0x5a, 0x53, 0x58, 0x31, 0x76, 0x4f, 0x79, 0x47, 0x63, 0x36,
		0x68, 0x50, 0x78, 0x65, 0x43, 0x46, 0x34, 0x39, 0x50, 0x6d,
		0x70, 0x75, 0x33, 0x74, 0x42, 0x67, 0x5a, 0x4b, 0x37, 0x53,
		0x68, 0x76, 0x4f, 0x30, 0x6c, 0x65, 0x65, 0x6a, 0x53, 0x61,
		0x4e, 0x6e, 0x4b, 0x61, 0x5a, 0x39, 0x49, 0x6c, 0x38, 0x66,
		0x52, 0x6a, 0x0a, 0x64, 0x4c, 0x71, 0x70, 0x44, 0x6b, 0x50,
		0x39, 0x37, 0x2b, 0x55, 0x48, 0x46, 0x4c, 0x42, 0x69, 0x56,
		0x65, 0x2b, 0x34, 0x50, 0x4b, 0x72, 0x55, 0x77, 0x4f, 0x72,
		0x64, 0x62, 0x35, 0x68, 0x6f, 0x48, 0x42, 0x6b, 0x38, 0x78,
		0x64, 0x73, 0x74, 0x4d, 0x34, 0x41, 0x74, 0x4f, 0x72, 0x33,
		0x2f, 0x51, 0x61, 0x38, 0x6c, 0x50, 0x79, 0x6c, 0x46, 0x74,
		0x47, 0x4e, 0x79, 0x50, 0x75, 0x77, 0x70, 0x0a, 0x38, 0x76,
		0x73, 0x64, 0x76, 0x72, 0x32, 0x37, 0x34, 0x68, 0x33, 0x31,
		0x74, 0x6d, 0x6c, 0x6a, 0x4c, 0x34, 0x65, 0x34, 0x6e, 0x52,
		0x39, 0x41, 0x76, 0x73, 0x44, 0x42, 0x6b, 0x37, 0x36, 0x76,
		0x38, 0x52, 0x5a, 0x6c, 0x79, 0x48, 0x54, 0x4e, 0x66, 0x54,
		0x32, 0x46, 0x69, 0x64, 0x73, 0x39, 0x7a, 0x6c, 0x61, 0x4c,
		0x7a, 0x55, 0x7a, 0x6b, 0x33, 0x7a, 0x52, 0x56, 0x2f, 0x2b,
		0x76, 0x69, 0x0a, 0x2f, 0x58, 0x37, 0x65, 0x75, 0x6c, 0x6c,
		0x48, 0x68, 0x70, 0x77, 0x6d, 0x63, 0x49, 0x74, 0x58, 0x34,
		0x65, 0x49, 0x4c, 0x52, 0x6c, 0x34, 0x75, 0x4b, 0x34, 0x72,
		0x37, 0x64, 0x74, 0x58, 0x70, 0x38, 0x6d, 0x71, 0x6b, 0x52,
		0x50, 0x46, 0x59, 0x2f, 0x68, 0x37, 0x6d, 0x66, 0x37, 0x55,
		0x43, 0x67, 0x67, 0x45, 0x42, 0x41, 0x4f, 0x61, 0x4d, 0x73,
		0x45, 0x4f, 0x79, 0x68, 0x66, 0x47, 0x42, 0x0a, 0x50, 0x4b,
		0x78, 0x37, 0x38, 0x67, 0x45, 0x6c, 0x47, 0x2f, 0x5a, 0x54,
		0x31, 0x74, 0x6f, 0x5a, 0x48, 0x73, 0x37, 0x4c, 0x6d, 0x77,
		0x75, 0x69, 0x68, 0x4b, 0x74, 0x59, 0x49, 0x7a, 0x38, 0x6e,
		0x53, 0x37, 0x45, 0x44, 0x63, 0x4c, 0x6e, 0x57, 0x55, 0x39,
		0x7a, 0x41, 0x49, 0x44, 0x47, 0x36, 0x31, 0x69, 0x63, 0x37,
		0x70, 0x36, 0x49, 0x6b, 0x77, 0x30, 0x73, 0x42, 0x67, 0x30,
		0x4c, 0x66, 0x0a, 0x6c, 0x72, 0x46, 0x4f, 0x46, 0x6d, 0x2f,
		0x64, 0x47, 0x6b, 0x45, 0x33, 0x6e, 0x62, 0x61, 0x41, 0x4c,
		0x33, 0x53, 0x51, 0x7a, 0x57, 0x41, 0x62, 0x35, 0x66, 0x53,
		0x75, 0x58, 0x4e, 0x38, 0x56, 0x6b, 0x45, 0x2b, 0x73, 0x6b,
		0x31, 0x5a, 0x57, 0x43, 0x74, 0x52, 0x31, 0x41, 0x68, 0x6b,
		0x6f, 0x58, 0x6f, 0x46, 0x46, 0x37, 0x35, 0x70, 0x68, 0x6a,
		0x4c, 0x37, 0x56, 0x2b, 0x35, 0x32, 0x44, 0x0a, 0x35, 0x6e,
		0x51, 0x72, 0x70, 0x50, 0x6f, 0x43, 0x65, 0x34, 0x54, 0x35,
		0x56, 0x70, 0x6b, 0x73, 0x61, 0x74, 0x77, 0x34, 0x6d, 0x4b,
		0x7a, 0x5a, 0x4b, 0x50, 0x4a, 0x2b, 0x34, 0x43, 0x53, 0x77,
		0x59, 0x43, 0x6a, 0x54, 0x34, 0x2f, 0x6e, 0x64, 0x44, 0x6d,
		0x43, 0x52, 0x74, 0x30, 0x6d, 0x78, 0x34, 0x31, 0x51, 0x4b,
		0x35, 0x79, 0x48, 0x57, 0x76, 0x41, 0x4d, 0x72, 0x34, 0x6e,
		0x72, 0x65, 0x0a, 0x42, 0x74, 0x4f, 0x72, 0x6c, 0x34, 0x53,
		0x5a, 0x77, 0x55, 0x75, 0x79, 0x58, 0x56, 0x63, 0x6e, 0x36,
		0x70, 0x53, 0x32, 0x69, 0x6b, 0x45, 0x79, 0x36, 0x51, 0x4e,
		0x2b, 0x64, 0x4c, 0x4d, 0x4c, 0x2f, 0x6b, 0x33, 0x39, 0x64,
		0x48, 0x2b, 0x79, 0x63, 0x31, 0x34, 0x76, 0x61, 0x51, 0x69,
		0x39, 0x2f, 0x66, 0x53, 0x45, 0x50, 0x6f, 0x74, 0x75, 0x53,
		0x76, 0x66, 0x5a, 0x78, 0x71, 0x57, 0x2b, 0x0a, 0x39, 0x6a,
		0x7a, 0x33, 0x47, 0x59, 0x52, 0x46, 0x2f, 0x4a, 0x30, 0x6a,
		0x69, 0x50, 0x4e, 0x4a, 0x34, 0x33, 0x6e, 0x6a, 0x6e, 0x52,
		0x4e, 0x72, 0x74, 0x36, 0x2f, 0x41, 0x6e, 0x76, 0x55, 0x70,
		0x68, 0x47, 0x78, 0x53, 0x72, 0x66, 0x36, 0x74, 0x76, 0x64,
		0x6f, 0x45, 0x30, 0x56, 0x32, 0x52, 0x77, 0x42, 0x36, 0x5a,
		0x75, 0x61, 0x64, 0x75, 0x32, 0x5a, 0x48, 0x48, 0x58, 0x49,
		0x6a, 0x73, 0x0a, 0x43, 0x43, 0x4a, 0x69, 0x69, 0x74, 0x56,
		0x2f, 0x52, 0x6a, 0x30, 0x43, 0x67, 0x67, 0x45, 0x42, 0x41,
		0x4b, 0x6a, 0x69, 0x50, 0x2f, 0x75, 0x52, 0x42, 0x6a, 0x33,
		0x44, 0x6e, 0x43, 0x2b, 0x67, 0x31, 0x52, 0x6c, 0x69, 0x45,
		0x56, 0x42, 0x6e, 0x52, 0x4f, 0x78, 0x53, 0x5a, 0x35, 0x6d,
		0x63, 0x57, 0x69, 0x6b, 0x41, 0x37, 0x52, 0x62, 0x73, 0x70,
		0x4c, 0x51, 0x73, 0x45, 0x72, 0x70, 0x37, 0x0a, 0x61, 0x70,
		0x47, 0x34, 0x66, 0x67, 0x62, 0x33, 0x37, 0x48, 0x53, 0x43,
		0x53, 0x52, 0x62, 0x59, 0x64, 0x70, 0x6a, 0x35, 0x2b, 0x34,
		0x61, 0x5a, 0x78, 0x51, 0x4b, 0x51, 0x4d, 0x7a, 0x4c, 0x65,
		0x6e, 0x2b, 0x5a, 0x63, 0x45, 0x4c, 0x49, 0x4c, 0x5a, 0x78,
		0x69, 0x45, 0x4d, 0x4a, 0x6c, 0x31, 0x35, 0x6e, 0x53, 0x79,
		0x74, 0x6e, 0x61, 0x70, 0x57, 0x72, 0x37, 0x6a, 0x37, 0x75,
		0x66, 0x6b, 0x0a, 0x6d, 0x6a, 0x46, 0x50, 0x38, 0x6e, 0x2f,
		0x63, 0x46, 0x67, 0x6a, 0x64, 0x4e, 0x44, 0x45, 0x50, 0x2f,
		0x36, 0x77, 0x4d, 0x31, 0x76, 0x59, 0x44, 0x6d, 0x44, 0x38,
		0x76, 0x52, 0x38, 0x47, 0x45, 0x58, 0x75, 0x33, 0x50, 0x6c,
		0x49, 0x33, 0x62, 0x72, 0x31, 0x70, 0x61, 0x73, 0x70, 0x4d,
		0x4f, 0x4c, 0x47, 0x55, 0x70, 0x35, 0x56, 0x45, 0x5a, 0x43,
		0x79, 0x56, 0x6a, 0x4f, 0x43, 0x55, 0x50, 0x0a, 0x56, 0x59,
		0x4c, 0x58, 0x63, 0x2b, 0x42, 0x72, 0x5a, 0x48, 0x4a, 0x51,
		0x39, 0x2b, 0x47, 0x51, 0x71, 0x53, 0x42, 0x5a, 0x53, 0x6f,
		0x78, 0x2f, 0x37, 0x59, 0x65, 0x38, 0x2b, 0x2f, 0x6d, 0x74,
		0x4f, 0x46, 0x5a, 0x62, 0x53, 0x4a, 0x6e, 0x45, 0x31, 0x41,
		0x6a, 0x46, 0x2f, 0x66, 0x65, 0x47, 0x6b, 0x4d, 0x30, 0x4a,
		0x5a, 0x36, 0x44, 0x31, 0x73, 0x4d, 0x6a, 0x75, 0x47, 0x61,
		0x79, 0x6f, 0x0a, 0x64, 0x61, 0x54, 0x33, 0x45, 0x35, 0x57,
		0x63, 0x35, 0x4e, 0x5a, 0x7a, 0x36, 0x75, 0x68, 0x4b, 0x67,
		0x41, 0x4f, 0x39, 0x47, 0x39, 0x2f, 0x6c, 0x6b, 0x47, 0x30,
		0x38, 0x62, 0x65, 0x2b, 0x47, 0x38, 0x75, 0x47, 0x36, 0x4a,
		0x6e, 0x45, 0x38, 0x59, 0x59, 0x74, 0x67, 0x51, 0x38, 0x64,
		0x33, 0x45, 0x38, 0x56, 0x6c, 0x53, 0x35, 0x77, 0x79, 0x65,
		0x4c, 0x42, 0x57, 0x5a, 0x67, 0x52, 0x2b, 0x0a, 0x45, 0x6e,
		0x6b, 0x33, 0x68, 0x79, 0x69, 0x6c, 0x6f, 0x4f, 0x71, 0x66,
		0x4c, 0x45, 0x71, 0x42, 0x68, 0x42, 0x55, 0x56, 0x32, 0x75,
		0x73, 0x6e, 0x6a, 0x57, 0x42, 0x2b, 0x6e, 0x32, 0x4b, 0x68,
		0x45, 0x6a, 0x63, 0x55, 0x48, 0x2f, 0x30, 0x43, 0x67, 0x67,
		0x45, 0x42, 0x41, 0x4b, 0x79, 0x48, 0x6c, 0x46, 0x44, 0x56,
		0x42, 0x45, 0x2b, 0x61, 0x75, 0x32, 0x45, 0x49, 0x30, 0x43,
		0x33, 0x7a, 0x0a, 0x4b, 0x31, 0x33, 0x48, 0x6c, 0x2f, 0x31,
		0x57, 0x31, 0x79, 0x74, 0x43, 0x53, 0x79, 0x45, 0x46, 0x32,
		0x65, 0x71, 0x76, 0x35, 0x70, 0x43, 0x59, 0x51, 0x35, 0x69,
		0x51, 0x54, 0x6f, 0x65, 0x44, 0x30, 0x50, 0x5a, 0x65, 0x41,
		0x4c, 0x6d, 0x32, 0x62, 0x42, 0x53, 0x6f, 0x47, 0x2b, 0x62,
		0x48, 0x52, 0x43, 0x36, 0x5a, 0x54, 0x7a, 0x56, 0x34, 0x52,
		0x54, 0x6b, 0x69, 0x69, 0x72, 0x2f, 0x6a, 0x0a, 0x34, 0x6f,
		0x58, 0x7a, 0x72, 0x4e, 0x36, 0x71, 0x52, 0x74, 0x6b, 0x67,
		0x6b, 0x39, 0x44, 0x63, 0x6a, 0x39, 0x58, 0x76, 0x74, 0x69,
		0x68, 0x72, 0x35, 0x72, 0x62, 0x35, 0x47, 0x72, 0x6c, 0x39,
		0x59, 0x6c, 0x56, 0x6a, 0x79, 0x74, 0x49, 0x77, 0x31, 0x36,
		0x67, 0x2b, 0x51, 0x66, 0x52, 0x79, 0x45, 0x2f, 0x57, 0x73,
		0x42, 0x4d, 0x50, 0x33, 0x74, 0x54, 0x6f, 0x79, 0x66, 0x78,
		0x39, 0x70, 0x0a, 0x52, 0x43, 0x35, 0x61, 0x75, 0x57, 0x65,
		0x66, 0x53, 0x63, 0x6a, 0x4c, 0x77, 0x64, 0x56, 0x34, 0x48,
		0x61, 0x42, 0x37, 0x4f, 0x68, 0x6d, 0x36, 0x73, 0x2f, 0x74,
		0x5a, 0x74, 0x6a, 0x32, 0x43, 0x4a, 0x41, 0x51, 0x34, 0x68,
		0x38, 0x56, 0x71, 0x6a, 0x67, 0x69, 0x53, 0x73, 0x43, 0x30,
		0x75, 0x6d, 0x54, 0x78, 0x32, 0x56, 0x75, 0x4a, 0x44, 0x7a,
		0x54, 0x41, 0x53, 0x4e, 0x53, 0x31, 0x53, 0x0a, 0x4d, 0x73,
		0x6b, 0x48, 0x44, 0x62, 0x72, 0x76, 0x63, 0x43, 0x6b, 0x69,
		0x41, 0x59, 0x79, 0x52, 0x49, 0x4c, 0x63, 0x77, 0x70, 0x76,
		0x78, 0x6d, 0x65, 0x55, 0x53, 0x37, 0x36, 0x37, 0x61, 0x30,
		0x51, 0x49, 0x6d, 0x33, 0x53, 0x51, 0x37, 0x31, 0x76, 0x31,
		0x71, 0x41, 0x72, 0x4a, 0x32, 0x44, 0x53, 0x43, 0x37, 0x36,
		0x6c, 0x69, 0x48, 0x72, 0x4b, 0x6a, 0x4b, 0x74, 0x61, 0x44,
		0x63, 0x49, 0x0a, 0x65, 0x4f, 0x62, 0x50, 0x64, 0x4f, 0x6b,
		0x37, 0x33, 0x6a, 0x4f, 0x56, 0x46, 0x6a, 0x4d, 0x32, 0x5a,
		0x52, 0x76, 0x74, 0x4e, 0x67, 0x65, 0x5a, 0x72, 0x6a, 0x4f,
		0x79, 0x47, 0x2b, 0x74, 0x2b, 0x38, 0x5a, 0x68, 0x36, 0x43,
		0x6f, 0x50, 0x50, 0x55, 0x73, 0x71, 0x54, 0x6c, 0x38, 0x2b,
		0x6f, 0x73, 0x4d, 0x44, 0x62, 0x52, 0x48, 0x42, 0x69, 0x30,
		0x69, 0x4f, 0x4a, 0x51, 0x77, 0x52, 0x4b, 0x0a, 0x58, 0x72,
		0x45, 0x43, 0x67, 0x67, 0x45, 0x41, 0x64, 0x49, 0x34, 0x75,
		0x44, 0x53, 0x34, 0x33, 0x74, 0x68, 0x2f, 0x34, 0x35, 0x67,
		0x41, 0x35, 0x42, 0x37, 0x42, 0x39, 0x36, 0x6a, 0x49, 0x4c,
		0x33, 0x55, 0x42, 0x2f, 0x31, 0x5a, 0x44, 0x31, 0x69, 0x61,
		0x45, 0x56, 0x5a, 0x79, 0x44, 0x39, 0x78, 0x73, 0x56, 0x33,
		0x42, 0x78, 0x6a, 0x70, 0x67, 0x41, 0x4b, 0x78, 0x31, 0x4a,
		0x41, 0x56, 0x0a, 0x56, 0x4e, 0x46, 0x30, 0x49, 0x59, 0x54,
		0x34, 0x6f, 0x61, 0x50, 0x35, 0x35, 0x72, 0x68, 0x7a, 0x6a,
		0x6f, 0x2b, 0x51, 0x2f, 0x41, 0x57, 0x4c, 0x52, 0x48, 0x4c,
		0x50, 0x54, 0x39, 0x64, 0x78, 0x67, 0x67, 0x4f, 0x53, 0x64,
		0x4d, 0x6d, 0x70, 0x4d, 0x33, 0x4d, 0x75, 0x63, 0x45, 0x36,
		0x44, 0x58, 0x6c, 0x69, 0x68, 0x43, 0x38, 0x42, 0x78, 0x62,
		0x72, 0x4d, 0x2f, 0x70, 0x4a, 0x63, 0x37, 0x0a, 0x69, 0x74,
		0x75, 0x30, 0x37, 0x64, 0x4b, 0x77, 0x66, 0x76, 0x74, 0x76,
		0x63, 0x64, 0x65, 0x73, 0x49, 0x72, 0x47, 0x32, 0x47, 0x33,
		0x45, 0x69, 0x38, 0x53, 0x46, 0x47, 0x34, 0x36, 0x6e, 0x6c,
		0x58, 0x76, 0x62, 0x59, 0x79, 0x75, 0x4c, 0x6c, 0x61, 0x73,
		0x56, 0x2f, 0x62, 0x67, 0x59, 0x52, 0x64, 0x7a, 0x2b, 0x46,
		0x6f, 0x61, 0x48, 0x4c, 0x7a, 0x63, 0x66, 0x55, 0x4f, 0x49,
		0x6b, 0x57, 0x0a, 0x52, 0x36, 0x4b, 0x66, 0x36, 0x51, 0x32,
		0x52, 0x45, 0x4e, 0x32, 0x6d, 0x74, 0x54, 0x54, 0x78, 0x62,
		0x32, 0x56, 0x78, 0x52, 0x38, 0x61, 0x64, 0x52, 0x67, 0x65,
		0x61, 0x71, 0x44, 0x54, 0x35, 0x57, 0x61, 0x49, 0x52, 0x47,
		0x62, 0x32, 0x4f, 0x54, 0x63, 0x45, 0x54, 0x5a, 0x4d, 0x41,
		0x38, 0x62, 0x42, 0x4a, 0x66, 0x52, 0x63, 0x2f, 0x41, 0x54,
		0x4b, 0x35, 0x6a, 0x73, 0x4d, 0x39, 0x30, 0x0a, 0x68, 0x39,
		0x62, 0x30, 0x2f, 0x56, 0x38, 0x47, 0x4c, 0x39, 0x38, 0x63,
		0x66, 0x46, 0x7a, 0x59, 0x53, 0x2b, 0x4b, 0x44, 0x79, 0x78,
		0x34, 0x52, 0x4a, 0x49, 0x39, 0x63, 0x6f, 0x42, 0x6a, 0x4a,
		0x69, 0x41, 0x46, 0x65, 0x62, 0x4c, 0x46, 0x51, 0x56, 0x46,
		0x5a, 0x32, 0x55, 0x43, 0x74, 0x45, 0x48, 0x66, 0x79, 0x32,
		0x36, 0x77, 0x6f, 0x6f, 0x4a, 0x33, 0x35, 0x73, 0x55, 0x33,
		0x34, 0x37, 0x0a, 0x5a, 0x2b, 0x4d, 0x6a, 0x42, 0x45, 0x55,
		0x77, 0x76, 0x45, 0x44, 0x7a, 0x67, 0x6f, 0x33, 0x6d, 0x32,
		0x68, 0x6b, 0x58, 0x4e, 0x4b, 0x68, 0x2f, 0x48, 0x71, 0x64,
		0x72, 0x77, 0x67, 0x3d, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d,
		0x2d, 0x45, 0x4e, 0x44, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41,
		0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d,
		0x2d, 0x0a,
	}},
};
/*

RSASHA512 (1024b)

rsa1024-512.	DNSKEY	  256  3  10  AwEAAbCq8MieaGKV/KQyex5Sft1IlyYm1VMafAF2qPUAc+wr2IYJbnX4AJ60pLVFpP2AfNHgbYaq4lkHvFyjl0TvyZQZCAr92lFpcPvVtWXKN+iA53kzpLwWihEBR4K5XogSZOyk9FZ/gb1oMO4FNLgIlKvBA0pQZ4lfzonuO139wGdR
rsa1024-512.	DS    	50388  10  1  96B0C31DDC0AAACDFF381BD0D8D7B59DD249E5A0
rsa1024-512.	DS    	50388  10  2  EB4D061FEC7F25A62DAC75627C686E97E3CF0DDB1852B203E97262F1978021C0
rsa1024-512.	DS    	50388  10  4  A1E822645D8BE029A321996B2DACA375907C8280D972B0E0499F3CCF4E83D5552AB222C9A0F1AA9968E05B51078AEEE0

-----BEGIN PRIVATE KEY-----
MIICdwIBADANBgkqhkiG9w0BAQEFAASCAmEwggJdAgEAAoGBALCq8MieaGKV/KQy
ex5Sft1IlyYm1VMafAF2qPUAc+wr2IYJbnX4AJ60pLVFpP2AfNHgbYaq4lkHvFyj
l0TvyZQZCAr92lFpcPvVtWXKN+iA53kzpLwWihEBR4K5XogSZOyk9FZ/gb1oMO4F
NLgIlKvBA0pQZ4lfzonuO139wGdRAgMBAAECgYEApIYMxkf/HhBfvunkDlD1JE49
tKHleAbqBNLkdYV+bypF8k1GOUZEDYhc+xNom0vU8f73ALqTICY8HGmcre5A4cUi
YIoCQweKu3OD2m2Trso2J1pT8+nUTvbyxgA5MrTF++DecpBR8fNe1uRzxIKOfXhE
jlaTUWta58M7fidv8gECQQDEAcp2YhmdIh+LzrWW96lAcR40Y7Kd3GPlm/U5zFKu
dthsv4BgdjzHtzHmQfRFyvZzztZCbU6+q60aJGcW2ss5AkEA5r3QZk3qgrqQMM3U
8C4bwhpFUVok9pSKD9FIXCw4FVaruv+ir1FKLSxYRwlpO77NZm4+Lw6QZ5G4F5/m
v8dE2QJBAJf2M3vYzrbuj7qaxlqHJGdby+keKosNJT4MzfDDoo60gVK+nc8rnC/S
rhMwqyu5EZ3zjFXv3Q8ilpAgDPTBKLECQAb4nR86I2ens0uGLHTFL9mBaa63hn5c
pn8r13/RZjBe2flgV83sKs2KlXAv/Jntfc8mbYKGch5u7JzqG9KXdqECQEolvGvG
uhKCZ2VM3cJsuAZhsIRKC1mbhbJSftv9DRQfHIIBjVNKKm+rPAje5xhkSrZ4WL3i
4XWXX6An/P++sl4=
-----END PRIVATE KEY-----

*/
static const key_parameters_t SAMPLE_RSA1024_SHA512_KEY = {
	.name = (uint8_t *)"\x0b""rsa1024-512",
	.flags = 256,
	.protocol = 3,
	.algorithm = 10,
	.public_key = { .size = 132, .data = (uint8_t []){
		0x03, 0x01, 0x00, 0x01, 0xb0, 0xaa, 0xf0, 0xc8, 0x9e, 0x68,
		0x62, 0x95, 0xfc, 0xa4, 0x32, 0x7b, 0x1e, 0x52, 0x7e, 0xdd,
		0x48, 0x97, 0x26, 0x26, 0xd5, 0x53, 0x1a, 0x7c, 0x01, 0x76,
		0xa8, 0xf5, 0x00, 0x73, 0xec, 0x2b, 0xd8, 0x86, 0x09, 0x6e,
		0x75, 0xf8, 0x00, 0x9e, 0xb4, 0xa4, 0xb5, 0x45, 0xa4, 0xfd,
		0x80, 0x7c, 0xd1, 0xe0, 0x6d, 0x86, 0xaa, 0xe2, 0x59, 0x07,
		0xbc, 0x5c, 0xa3, 0x97, 0x44, 0xef, 0xc9, 0x94, 0x19, 0x08,
		0x0a, 0xfd, 0xda, 0x51, 0x69, 0x70, 0xfb, 0xd5, 0xb5, 0x65,
		0xca, 0x37, 0xe8, 0x80, 0xe7, 0x79, 0x33, 0xa4, 0xbc, 0x16,
		0x8a, 0x11, 0x01, 0x47, 0x82, 0xb9, 0x5e, 0x88, 0x12, 0x64,
		0xec, 0xa4, 0xf4, 0x56, 0x7f, 0x81, 0xbd, 0x68, 0x30, 0xee,
		0x05, 0x34, 0xb8, 0x08, 0x94, 0xab, 0xc1, 0x03, 0x4a, 0x50,
		0x67, 0x89, 0x5f, 0xce, 0x89, 0xee, 0x3b, 0x5d, 0xfd, 0xc0,
		0x67, 0x51,
	}},
	.rdata = { .size = 136, .data = (uint8_t []){
		01, 0x00, 0x03, 0x0a,
		0x03, 0x01, 0x00, 0x01, 0xb0, 0xaa, 0xf0, 0xc8, 0x9e, 0x68,
		0x62, 0x95, 0xfc, 0xa4, 0x32, 0x7b, 0x1e, 0x52, 0x7e, 0xdd,
		0x48, 0x97, 0x26, 0x26, 0xd5, 0x53, 0x1a, 0x7c, 0x01, 0x76,
		0xa8, 0xf5, 0x00, 0x73, 0xec, 0x2b, 0xd8, 0x86, 0x09, 0x6e,
		0x75, 0xf8, 0x00, 0x9e, 0xb4, 0xa4, 0xb5, 0x45, 0xa4, 0xfd,
		0x80, 0x7c, 0xd1, 0xe0, 0x6d, 0x86, 0xaa, 0xe2, 0x59, 0x07,
		0xbc, 0x5c, 0xa3, 0x97, 0x44, 0xef, 0xc9, 0x94, 0x19, 0x08,
		0x0a, 0xfd, 0xda, 0x51, 0x69, 0x70, 0xfb, 0xd5, 0xb5, 0x65,
		0xca, 0x37, 0xe8, 0x80, 0xe7, 0x79, 0x33, 0xa4, 0xbc, 0x16,
		0x8a, 0x11, 0x01, 0x47, 0x82, 0xb9, 0x5e, 0x88, 0x12, 0x64,
		0xec, 0xa4, 0xf4, 0x56, 0x7f, 0x81, 0xbd, 0x68, 0x30, 0xee,
		0x05, 0x34, 0xb8, 0x08, 0x94, 0xab, 0xc1, 0x03, 0x4a, 0x50,
		0x67, 0x89, 0x5f, 0xce, 0x89, 0xee, 0x3b, 0x5d, 0xfd, 0xc0,
		0x67, 0x51,
	}},
	.key_id = "03f1a686d8f3b377137f45978467c30b524d1e32",
	.keytag = 50388,
	.ds_sha1 = { .size = 24, .data = (uint8_t []){
		0xc4, 0xd4, 0x0a, 0x01, 0x96, 0xb0, 0xc3, 0x1d, 0xdc, 0x0a,
		0xaa, 0xcd, 0xff, 0x38, 0x1b, 0xd0, 0xd8, 0xd7, 0xb5, 0x9d,
		0xd2, 0x49, 0xe5, 0xa0,
	}},
	.ds_sha256 = { .size = 36, .data = (uint8_t []){
		0xc4, 0xd4, 0x0a, 0x02, 0xeb, 0x4d, 0x06, 0x1f, 0xec, 0x7f,
		0x25, 0xa6, 0x2d, 0xac, 0x75, 0x62, 0x7c, 0x68, 0x6e, 0x97,
		0xe3, 0xcf, 0x0d, 0xdb, 0x18, 0x52, 0xb2, 0x03, 0xe9, 0x72,
		0x62, 0xf1, 0x97, 0x80, 0x21, 0xc0,
	}},
	.ds_sha384 = { .size = 52, .data = (uint8_t []){
		0xc4, 0xd4, 0x0a, 0x04, 0xa1, 0xe8, 0x22, 0x64, 0x5d, 0x8b,
		0xe0, 0x29, 0xa3, 0x21, 0x99, 0x6b, 0x2d, 0xac, 0xa3, 0x75,
		0x90, 0x7c, 0x82, 0x80, 0xd9, 0x72, 0xb0, 0xe0, 0x49, 0x9f,
		0x3c, 0xcf, 0x4e, 0x83, 0xd5, 0x55, 0x2a, 0xb2, 0x22, 0xc9,
		0xa0, 0xf1, 0xaa, 0x99, 0x68, 0xe0, 0x5b, 0x51, 0x07, 0x8a,
		0xee, 0xe0,
	}},
	.bit_size = 1024,
	.pem = { .size = 916, .data = (uint8_t []){
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49,
		0x49, 0x43, 0x64, 0x77, 0x49, 0x42, 0x41, 0x44, 0x41, 0x4e,
		0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77,
		0x30, 0x42, 0x41, 0x51, 0x45, 0x46, 0x41, 0x41, 0x53, 0x43,
		0x41, 0x6d, 0x45, 0x77, 0x67, 0x67, 0x4a, 0x64, 0x41, 0x67,
		0x45, 0x41, 0x41, 0x6f, 0x47, 0x42, 0x41, 0x4c, 0x43, 0x71,
		0x38, 0x4d, 0x69, 0x65, 0x61, 0x47, 0x4b, 0x56, 0x2f, 0x4b,
		0x51, 0x79, 0x0a, 0x65, 0x78, 0x35, 0x53, 0x66, 0x74, 0x31,
		0x49, 0x6c, 0x79, 0x59, 0x6d, 0x31, 0x56, 0x4d, 0x61, 0x66,
		0x41, 0x46, 0x32, 0x71, 0x50, 0x55, 0x41, 0x63, 0x2b, 0x77,
		0x72, 0x32, 0x49, 0x59, 0x4a, 0x62, 0x6e, 0x58, 0x34, 0x41,
		0x4a, 0x36, 0x30, 0x70, 0x4c, 0x56, 0x46, 0x70, 0x50, 0x32,
		0x41, 0x66, 0x4e, 0x48, 0x67, 0x62, 0x59, 0x61, 0x71, 0x34,
		0x6c, 0x6b, 0x48, 0x76, 0x46, 0x79, 0x6a, 0x0a, 0x6c, 0x30,
		0x54, 0x76, 0x79, 0x5a, 0x51, 0x5a, 0x43, 0x41, 0x72, 0x39,
		0x32, 0x6c, 0x46, 0x70, 0x63, 0x50, 0x76, 0x56, 0x74, 0x57,
		0x58, 0x4b, 0x4e, 0x2b, 0x69, 0x41, 0x35, 0x33, 0x6b, 0x7a,
		0x70, 0x4c, 0x77, 0x57, 0x69, 0x68, 0x45, 0x42, 0x52, 0x34,
		0x4b, 0x35, 0x58, 0x6f, 0x67, 0x53, 0x5a, 0x4f, 0x79, 0x6b,
		0x39, 0x46, 0x5a, 0x2f, 0x67, 0x62, 0x31, 0x6f, 0x4d, 0x4f,
		0x34, 0x46, 0x0a, 0x4e, 0x4c, 0x67, 0x49, 0x6c, 0x4b, 0x76,
		0x42, 0x41, 0x30, 0x70, 0x51, 0x5a, 0x34, 0x6c, 0x66, 0x7a,
		0x6f, 0x6e, 0x75, 0x4f, 0x31, 0x33, 0x39, 0x77, 0x47, 0x64,
		0x52, 0x41, 0x67, 0x4d, 0x42, 0x41, 0x41, 0x45, 0x43, 0x67,
		0x59, 0x45, 0x41, 0x70, 0x49, 0x59, 0x4d, 0x78, 0x6b, 0x66,
		0x2f, 0x48, 0x68, 0x42, 0x66, 0x76, 0x75, 0x6e, 0x6b, 0x44,
		0x6c, 0x44, 0x31, 0x4a, 0x45, 0x34, 0x39, 0x0a, 0x74, 0x4b,
		0x48, 0x6c, 0x65, 0x41, 0x62, 0x71, 0x42, 0x4e, 0x4c, 0x6b,
		0x64, 0x59, 0x56, 0x2b, 0x62, 0x79, 0x70, 0x46, 0x38, 0x6b,
		0x31, 0x47, 0x4f, 0x55, 0x5a, 0x45, 0x44, 0x59, 0x68, 0x63,
		0x2b, 0x78, 0x4e, 0x6f, 0x6d, 0x30, 0x76, 0x55, 0x38, 0x66,
		0x37, 0x33, 0x41, 0x4c, 0x71, 0x54, 0x49, 0x43, 0x59, 0x38,
		0x48, 0x47, 0x6d, 0x63, 0x72, 0x65, 0x35, 0x41, 0x34, 0x63,
		0x55, 0x69, 0x0a, 0x59, 0x49, 0x6f, 0x43, 0x51, 0x77, 0x65,
		0x4b, 0x75, 0x33, 0x4f, 0x44, 0x32, 0x6d, 0x32, 0x54, 0x72,
		0x73, 0x6f, 0x32, 0x4a, 0x31, 0x70, 0x54, 0x38, 0x2b, 0x6e,
		0x55, 0x54, 0x76, 0x62, 0x79, 0x78, 0x67, 0x41, 0x35, 0x4d,
		0x72, 0x54, 0x46, 0x2b, 0x2b, 0x44, 0x65, 0x63, 0x70, 0x42,
		0x52, 0x38, 0x66, 0x4e, 0x65, 0x31, 0x75, 0x52, 0x7a, 0x78,
		0x49, 0x4b, 0x4f, 0x66, 0x58, 0x68, 0x45, 0x0a, 0x6a, 0x6c,
		0x61, 0x54, 0x55, 0x57, 0x74, 0x61, 0x35, 0x38, 0x4d, 0x37,
		0x66, 0x69, 0x64, 0x76, 0x38, 0x67, 0x45, 0x43, 0x51, 0x51,
		0x44, 0x45, 0x41, 0x63, 0x70, 0x32, 0x59, 0x68, 0x6d, 0x64,
		0x49, 0x68, 0x2b, 0x4c, 0x7a, 0x72, 0x57, 0x57, 0x39, 0x36,
		0x6c, 0x41, 0x63, 0x52, 0x34, 0x30, 0x59, 0x37, 0x4b, 0x64,
		0x33, 0x47, 0x50, 0x6c, 0x6d, 0x2f, 0x55, 0x35, 0x7a, 0x46,
		0x4b, 0x75, 0x0a, 0x64, 0x74, 0x68, 0x73, 0x76, 0x34, 0x42,
		0x67, 0x64, 0x6a, 0x7a, 0x48, 0x74, 0x7a, 0x48, 0x6d, 0x51,
		0x66, 0x52, 0x46, 0x79, 0x76, 0x5a, 0x7a, 0x7a, 0x74, 0x5a,
		0x43, 0x62, 0x55, 0x36, 0x2b, 0x71, 0x36, 0x30, 0x61, 0x4a,
		0x47, 0x63, 0x57, 0x32, 0x73, 0x73, 0x35, 0x41, 0x6b, 0x45,
		0x41, 0x35, 0x72, 0x33, 0x51, 0x5a, 0x6b, 0x33, 0x71, 0x67,
		0x72, 0x71, 0x51, 0x4d, 0x4d, 0x33, 0x55, 0x0a, 0x38, 0x43,
		0x34, 0x62, 0x77, 0x68, 0x70, 0x46, 0x55, 0x56, 0x6f, 0x6b,
		0x39, 0x70, 0x53, 0x4b, 0x44, 0x39, 0x46, 0x49, 0x58, 0x43,
		0x77, 0x34, 0x46, 0x56, 0x61, 0x72, 0x75, 0x76, 0x2b, 0x69,
		0x72, 0x31, 0x46, 0x4b, 0x4c, 0x53, 0x78, 0x59, 0x52, 0x77,
		0x6c, 0x70, 0x4f, 0x37, 0x37, 0x4e, 0x5a, 0x6d, 0x34, 0x2b,
		0x4c, 0x77, 0x36, 0x51, 0x5a, 0x35, 0x47, 0x34, 0x46, 0x35,
		0x2f, 0x6d, 0x0a, 0x76, 0x38, 0x64, 0x45, 0x32, 0x51, 0x4a,
		0x42, 0x41, 0x4a, 0x66, 0x32, 0x4d, 0x33, 0x76, 0x59, 0x7a,
		0x72, 0x62, 0x75, 0x6a, 0x37, 0x71, 0x61, 0x78, 0x6c, 0x71,
		0x48, 0x4a, 0x47, 0x64, 0x62, 0x79, 0x2b, 0x6b, 0x65, 0x4b,
		0x6f, 0x73, 0x4e, 0x4a, 0x54, 0x34, 0x4d, 0x7a, 0x66, 0x44,
		0x44, 0x6f, 0x6f, 0x36, 0x30, 0x67, 0x56, 0x4b, 0x2b, 0x6e,
		0x63, 0x38, 0x72, 0x6e, 0x43, 0x2f, 0x53, 0x0a, 0x72, 0x68,
		0x4d, 0x77, 0x71, 0x79, 0x75, 0x35, 0x45, 0x5a, 0x33, 0x7a,
		0x6a, 0x46, 0x58, 0x76, 0x33, 0x51, 0x38, 0x69, 0x6c, 0x70,
		0x41, 0x67, 0x44, 0x50, 0x54, 0x42, 0x4b, 0x4c, 0x45, 0x43,
		0x51, 0x41, 0x62, 0x34, 0x6e, 0x52, 0x38, 0x36, 0x49, 0x32,
		0x65, 0x6e, 0x73, 0x30, 0x75, 0x47, 0x4c, 0x48, 0x54, 0x46,
		0x4c, 0x39, 0x6d, 0x42, 0x61, 0x61, 0x36, 0x33, 0x68, 0x6e,
		0x35, 0x63, 0x0a, 0x70, 0x6e, 0x38, 0x72, 0x31, 0x33, 0x2f,
		0x52, 0x5a, 0x6a, 0x42, 0x65, 0x32, 0x66, 0x6c, 0x67, 0x56,
		0x38, 0x33, 0x73, 0x4b, 0x73, 0x32, 0x4b, 0x6c, 0x58, 0x41,
		0x76, 0x2f, 0x4a, 0x6e, 0x74, 0x66, 0x63, 0x38, 0x6d, 0x62,
		0x59, 0x4b, 0x47, 0x63, 0x68, 0x35, 0x75, 0x37, 0x4a, 0x7a,
		0x71, 0x47, 0x39, 0x4b, 0x58, 0x64, 0x71, 0x45, 0x43, 0x51,
		0x45, 0x6f, 0x6c, 0x76, 0x47, 0x76, 0x47, 0x0a, 0x75, 0x68,
		0x4b, 0x43, 0x5a, 0x32, 0x56, 0x4d, 0x33, 0x63, 0x4a, 0x73,
		0x75, 0x41, 0x5a, 0x68, 0x73, 0x49, 0x52, 0x4b, 0x43, 0x31,
		0x6d, 0x62, 0x68, 0x62, 0x4a, 0x53, 0x66, 0x74, 0x76, 0x39,
		0x44, 0x52, 0x51, 0x66, 0x48, 0x49, 0x49, 0x42, 0x6a, 0x56,
		0x4e, 0x4b, 0x4b, 0x6d, 0x2b, 0x72, 0x50, 0x41, 0x6a, 0x65,
		0x35, 0x78, 0x68, 0x6b, 0x53, 0x72, 0x5a, 0x34, 0x57, 0x4c,
		0x33, 0x69, 0x0a, 0x34, 0x58, 0x57, 0x58, 0x58, 0x36, 0x41,
		0x6e, 0x2f, 0x50, 0x2b, 0x2b, 0x73, 0x6c, 0x34, 0x3d, 0x0a,
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x50,
		0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59,
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a,
	}},
};
/*

RSASHA512 (2048b)

rsa2048-512.	DNSKEY	  256  3  10  AwEAAcAt1xtWEpAqrnkLLEbzXP7uzdIkOepNrzsQp9XldmpXtdTpkz+m4IDZc0VoitO7OeVT13VwIeZ/JxckqtHov8/DLeSm5ExmfcGkjh8Trliles+FOEs3Zk4sXqunlULLU7FqO5SLdnjQ9qlV1wsfQk/EiSmtgyWcEYE6Yg0Z1Kb67fOAZhFhrl8V5g0VxNYDm2SF9pXeNoxpJ25WxbUf58fy7rFRAtYHOQf44smiu84ovhkasWfFIYcIES/DWe7en6CnyDrLkNLq4lZnPdeArMOt8Z6dipQWirKfz+JPQjiahlTwi99WEJHPJ4jrHd93VpYbJlSKV9odMtwhaTFl55U=
rsa2048-512.	DS    	52908  10  1  2D85820E982CF5DEE9A7F315F95D868F55366F03
rsa2048-512.	DS    	52908  10  2  9768222BD72C197F45144E2F35A75CDA94785951AFA160432683ADB073EB06E9
rsa2048-512.	DS    	52908  10  4  3CCF23DF0CF263DF8AC5D49285C95FAF3EBDD4E95C2E34BF7B4C01C935EBC32492202802B3E3C81CB5AE515EC576C682

-----BEGIN PRIVATE KEY-----
MIIEvQIBADANBgkqhkiG9w0BAQEFAASCBKcwggSjAgEAAoIBAQDALdcbVhKQKq55
CyxG81z+7s3SJDnqTa87EKfV5XZqV7XU6ZM/puCA2XNFaIrTuznlU9d1cCHmfycX
JKrR6L/Pwy3kpuRMZn3BpI4fE65YpXrPhThLN2ZOLF6rp5VCy1OxajuUi3Z40Pap
VdcLH0JPxIkprYMlnBGBOmINGdSm+u3zgGYRYa5fFeYNFcTWA5tkhfaV3jaMaSdu
VsW1H+fH8u6xUQLWBzkH+OLJorvOKL4ZGrFnxSGHCBEvw1nu3p+gp8g6y5DS6uJW
Zz3XgKzDrfGenYqUFoqyn8/iT0I4moZU8IvfVhCRzyeI6x3fd1aWGyZUilfaHTLc
IWkxZeeVAgMBAAECggEASYC+5zuyeeV5Od6mHrTx+qeHnTox1IXF4cdrCgprkhzd
9QMpSI0hDOceiYuEnZrcd/PrUoWC0dAMOXPSydpFksdKBEw4wWfp7Whs20IByvHv
2PNRJqOXw4qA4xKDqjpuVksVYqk8tNTnorisORrJmAzJkWFF6dqZb13PJsecaAij
aAU1ViCgNZgj7Khh1nOR6ZGa+RZmxnMOnNt3ERMWQeM27U9UvRQXU0h6+EzOziMY
YwCNklkCD3Cy5OczchXz6WYHJNuW6YGTB4o/dOELKv3KHBx+MeE64cRM5fsnv44H
e84DfYbZu+hVaJlkkGOdvVrSNW4isOy4s5I34E4xuQKBgQDO0paiXIhoES8voeaS
tdKr4IV2gZrHe73ZNaYQSGQf09wQFLIh6evUSE+7NYtv0JtL7gmXVYrzjjT5wE4X
HyC3j4K1cj5N9DXW9kXcebDjAbhn0PbPf1xAY8YujwVUJsVcIm3/s3NMkBuZRoNl
R0Ol+YajFGXKbp9Pt24OoKqSewKBgQDt3+QCyfg8s6kNGUsID1h/SHw8ZoaOKlN2
a7v22Azw0OvIXMlYCwDQqoBXOXuzPA6tKGGi0If1OUgReZvc2hApj0GKsiMMR8Do
EV/0pboN5hAKQ5CbMSqqsD//thwORjpTF417kdlrBd4cNgoJA6QnkCz/cENajFKM
Be32OiUZLwKBgAw0GtVLtnTFhs49tFg1yE+B+P4MGbDpmUMmyjytuNKz8ok0X2pm
3hZxidrXBvo7Cd7KFeDLGF7UpkWSVklA+fI60pCYRBrfbQnZEDg5KfnLETF3AAkw
T6Yw/PGkWMElZIHSOkFbOlNrMWWy5m/ka5ZwicQ3oOX37wiKH0FXs5jfAoGBAIy4
cHi95p5yYkWfeuWvvli+FtDcjik4HNvna/DhavZjgbisO2hcSgUneHkhRm6Hzt7d
DrV0y5lkY6AC4xPJ5NnRJ6LXMyimohYngS6HhAOI4GBSDA/jDCyhy5qIrxw6q3JQ
cqBxbb+EXxstVM50Wv4UqqoKTa4ZaRPwxg9t4lh9AoGAEOOZ4s0hOjoMLyZx6NXQ
biHJS2DNxjE4opzLvP6RFmqBtgp5n3wvqvBBLQKLguLti+G/UjiywTFl8Wj4PEhj
T9UObU8PuXF3XzWx3WkznYrFL59N1Q01+di31fXLFAi04C9LItxHrRsmYmgaPzoJ
nZPcvyLnplvFLPoT1EB32fM=
-----END PRIVATE KEY-----

*/
static const key_parameters_t SAMPLE_RSA2048_SHA512_KEY = {
	.name = (uint8_t *)"\x0b""rsa2048-512",
	.flags = 256,
	.protocol = 3,
	.algorithm = 10,
	.public_key = { .size = 260, .data = (uint8_t []){
		0x03, 0x01, 0x00, 0x01, 0xc0, 0x2d, 0xd7, 0x1b, 0x56, 0x12,
		0x90, 0x2a, 0xae, 0x79, 0x0b, 0x2c, 0x46, 0xf3, 0x5c, 0xfe,
		0xee, 0xcd, 0xd2, 0x24, 0x39, 0xea, 0x4d, 0xaf, 0x3b, 0x10,
		0xa7, 0xd5, 0xe5, 0x76, 0x6a, 0x57, 0xb5, 0xd4, 0xe9, 0x93,
		0x3f, 0xa6, 0xe0, 0x80, 0xd9, 0x73, 0x45, 0x68, 0x8a, 0xd3,
		0xbb, 0x39, 0xe5, 0x53, 0xd7, 0x75, 0x70, 0x21, 0xe6, 0x7f,
		0x27, 0x17, 0x24, 0xaa, 0xd1, 0xe8, 0xbf, 0xcf, 0xc3, 0x2d,
		0xe4, 0xa6, 0xe4, 0x4c, 0x66, 0x7d, 0xc1, 0xa4, 0x8e, 0x1f,
		0x13, 0xae, 0x58, 0xa5, 0x7a, 0xcf, 0x85, 0x38, 0x4b, 0x37,
		0x66, 0x4e, 0x2c, 0x5e, 0xab, 0xa7, 0x95, 0x42, 0xcb, 0x53,
		0xb1, 0x6a, 0x3b, 0x94, 0x8b, 0x76, 0x78, 0xd0, 0xf6, 0xa9,
		0x55, 0xd7, 0x0b, 0x1f, 0x42, 0x4f, 0xc4, 0x89, 0x29, 0xad,
		0x83, 0x25, 0x9c, 0x11, 0x81, 0x3a, 0x62, 0x0d, 0x19, 0xd4,
		0xa6, 0xfa, 0xed, 0xf3, 0x80, 0x66, 0x11, 0x61, 0xae, 0x5f,
		0x15, 0xe6, 0x0d, 0x15, 0xc4, 0xd6, 0x03, 0x9b, 0x64, 0x85,
		0xf6, 0x95, 0xde, 0x36, 0x8c, 0x69, 0x27, 0x6e, 0x56, 0xc5,
		0xb5, 0x1f, 0xe7, 0xc7, 0xf2, 0xee, 0xb1, 0x51, 0x02, 0xd6,
		0x07, 0x39, 0x07, 0xf8, 0xe2, 0xc9, 0xa2, 0xbb, 0xce, 0x28,
		0xbe, 0x19, 0x1a, 0xb1, 0x67, 0xc5, 0x21, 0x87, 0x08, 0x11,
		0x2f, 0xc3, 0x59, 0xee, 0xde, 0x9f, 0xa0, 0xa7, 0xc8, 0x3a,
		0xcb, 0x90, 0xd2, 0xea, 0xe2, 0x56, 0x67, 0x3d, 0xd7, 0x80,
		0xac, 0xc3, 0xad, 0xf1, 0x9e, 0x9d, 0x8a, 0x94, 0x16, 0x8a,
		0xb2, 0x9f, 0xcf, 0xe2, 0x4f, 0x42, 0x38, 0x9a, 0x86, 0x54,
		0xf0, 0x8b, 0xdf, 0x56, 0x10, 0x91, 0xcf, 0x27, 0x88, 0xeb,
		0x1d, 0xdf, 0x77, 0x56, 0x96, 0x1b, 0x26, 0x54, 0x8a, 0x57,
		0xda, 0x1d, 0x32, 0xdc, 0x21, 0x69, 0x31, 0x65, 0xe7, 0x95,
	}},
	.rdata = { .size = 264, .data = (uint8_t []){
		01, 0x00, 0x03, 0x0a,
		0x03, 0x01, 0x00, 0x01, 0xc0, 0x2d, 0xd7, 0x1b, 0x56, 0x12,
		0x90, 0x2a, 0xae, 0x79, 0x0b, 0x2c, 0x46, 0xf3, 0x5c, 0xfe,
		0xee, 0xcd, 0xd2, 0x24, 0x39, 0xea, 0x4d, 0xaf, 0x3b, 0x10,
		0xa7, 0xd5, 0xe5, 0x76, 0x6a, 0x57, 0xb5, 0xd4, 0xe9, 0x93,
		0x3f, 0xa6, 0xe0, 0x80, 0xd9, 0x73, 0x45, 0x68, 0x8a, 0xd3,
		0xbb, 0x39, 0xe5, 0x53, 0xd7, 0x75, 0x70, 0x21, 0xe6, 0x7f,
		0x27, 0x17, 0x24, 0xaa, 0xd1, 0xe8, 0xbf, 0xcf, 0xc3, 0x2d,
		0xe4, 0xa6, 0xe4, 0x4c, 0x66, 0x7d, 0xc1, 0xa4, 0x8e, 0x1f,
		0x13, 0xae, 0x58, 0xa5, 0x7a, 0xcf, 0x85, 0x38, 0x4b, 0x37,
		0x66, 0x4e, 0x2c, 0x5e, 0xab, 0xa7, 0x95, 0x42, 0xcb, 0x53,
		0xb1, 0x6a, 0x3b, 0x94, 0x8b, 0x76, 0x78, 0xd0, 0xf6, 0xa9,
		0x55, 0xd7, 0x0b, 0x1f, 0x42, 0x4f, 0xc4, 0x89, 0x29, 0xad,
		0x83, 0x25, 0x9c, 0x11, 0x81, 0x3a, 0x62, 0x0d, 0x19, 0xd4,
		0xa6, 0xfa, 0xed, 0xf3, 0x80, 0x66, 0x11, 0x61, 0xae, 0x5f,
		0x15, 0xe6, 0x0d, 0x15, 0xc4, 0xd6, 0x03, 0x9b, 0x64, 0x85,
		0xf6, 0x95, 0xde, 0x36, 0x8c, 0x69, 0x27, 0x6e, 0x56, 0xc5,
		0xb5, 0x1f, 0xe7, 0xc7, 0xf2, 0xee, 0xb1, 0x51, 0x02, 0xd6,
		0x07, 0x39, 0x07, 0xf8, 0xe2, 0xc9, 0xa2, 0xbb, 0xce, 0x28,
		0xbe, 0x19, 0x1a, 0xb1, 0x67, 0xc5, 0x21, 0x87, 0x08, 0x11,
		0x2f, 0xc3, 0x59, 0xee, 0xde, 0x9f, 0xa0, 0xa7, 0xc8, 0x3a,
		0xcb, 0x90, 0xd2, 0xea, 0xe2, 0x56, 0x67, 0x3d, 0xd7, 0x80,
		0xac, 0xc3, 0xad, 0xf1, 0x9e, 0x9d, 0x8a, 0x94, 0x16, 0x8a,
		0xb2, 0x9f, 0xcf, 0xe2, 0x4f, 0x42, 0x38, 0x9a, 0x86, 0x54,
		0xf0, 0x8b, 0xdf, 0x56, 0x10, 0x91, 0xcf, 0x27, 0x88, 0xeb,
		0x1d, 0xdf, 0x77, 0x56, 0x96, 0x1b, 0x26, 0x54, 0x8a, 0x57,
		0xda, 0x1d, 0x32, 0xdc, 0x21, 0x69, 0x31, 0x65, 0xe7, 0x95,
	}},
	.key_id = "a9cbfcd0519f92fc967091a896370fcb1485514e",
	.keytag = 52908,
	.ds_sha1 = { .size = 24, .data = (uint8_t []){
		0xce, 0xac, 0x0a, 0x01, 0x2d, 0x85, 0x82, 0x0e, 0x98, 0x2c,
		0xf5, 0xde, 0xe9, 0xa7, 0xf3, 0x15, 0xf9, 0x5d, 0x86, 0x8f,
		0x55, 0x36, 0x6f, 0x03,
	}},
	.ds_sha256 = { .size = 36, .data = (uint8_t []){
		0xce, 0xac, 0x0a, 0x02, 0x97, 0x68, 0x22, 0x2b, 0xd7, 0x2c,
		0x19, 0x7f, 0x45, 0x14, 0x4e, 0x2f, 0x35, 0xa7, 0x5c, 0xda,
		0x94, 0x78, 0x59, 0x51, 0xaf, 0xa1, 0x60, 0x43, 0x26, 0x83,
		0xad, 0xb0, 0x73, 0xeb, 0x06, 0xe9,
	}},
	.ds_sha384 = { .size = 52, .data = (uint8_t []){
		0xce, 0xac, 0x0a, 0x04, 0x3c, 0xcf, 0x23, 0xdf, 0x0c, 0xf2,
		0x63, 0xdf, 0x8a, 0xc5, 0xd4, 0x92, 0x85, 0xc9, 0x5f, 0xaf,
		0x3e, 0xbd, 0xd4, 0xe9, 0x5c, 0x2e, 0x34, 0xbf, 0x7b, 0x4c,
		0x01, 0xc9, 0x35, 0xeb, 0xc3, 0x24, 0x92, 0x20, 0x28, 0x02,
		0xb3, 0xe3, 0xc8, 0x1c, 0xb5, 0xae, 0x51, 0x5e, 0xc5, 0x76,
		0xc6, 0x82,
	}},
	.bit_size = 2048,
	.pem = { .size = 1704, .data = (uint8_t []){
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49,
		0x49, 0x45, 0x76, 0x51, 0x49, 0x42, 0x41, 0x44, 0x41, 0x4e,
		0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77,
		0x30, 0x42, 0x41, 0x51, 0x45, 0x46, 0x41, 0x41, 0x53, 0x43,
		0x42, 0x4b, 0x63, 0x77, 0x67, 0x67, 0x53, 0x6a, 0x41, 0x67,
		0x45, 0x41, 0x41, 0x6f, 0x49, 0x42, 0x41, 0x51, 0x44, 0x41,
		0x4c, 0x64, 0x63, 0x62, 0x56, 0x68, 0x4b, 0x51, 0x4b, 0x71,
		0x35, 0x35, 0x0a, 0x43, 0x79, 0x78, 0x47, 0x38, 0x31, 0x7a,
		0x2b, 0x37, 0x73, 0x33, 0x53, 0x4a, 0x44, 0x6e, 0x71, 0x54,
		0x61, 0x38, 0x37, 0x45, 0x4b, 0x66, 0x56, 0x35, 0x58, 0x5a,
		0x71, 0x56, 0x37, 0x58, 0x55, 0x36, 0x5a, 0x4d, 0x2f, 0x70,
		0x75, 0x43, 0x41, 0x32, 0x58, 0x4e, 0x46, 0x61, 0x49, 0x72,
		0x54, 0x75, 0x7a, 0x6e, 0x6c, 0x55, 0x39, 0x64, 0x31, 0x63,
		0x43, 0x48, 0x6d, 0x66, 0x79, 0x63, 0x58, 0x0a, 0x4a, 0x4b,
		0x72, 0x52, 0x36, 0x4c, 0x2f, 0x50, 0x77, 0x79, 0x33, 0x6b,
		0x70, 0x75, 0x52, 0x4d, 0x5a, 0x6e, 0x33, 0x42, 0x70, 0x49,
		0x34, 0x66, 0x45, 0x36, 0x35, 0x59, 0x70, 0x58, 0x72, 0x50,
		0x68, 0x54, 0x68, 0x4c, 0x4e, 0x32, 0x5a, 0x4f, 0x4c, 0x46,
		0x36, 0x72, 0x70, 0x35, 0x56, 0x43, 0x79, 0x31, 0x4f, 0x78,
		0x61, 0x6a, 0x75, 0x55, 0x69, 0x33, 0x5a, 0x34, 0x30, 0x50,
		0x61, 0x70, 0x0a, 0x56, 0x64, 0x63, 0x4c, 0x48, 0x30, 0x4a,
		0x50, 0x78, 0x49, 0x6b, 0x70, 0x72, 0x59, 0x4d, 0x6c, 0x6e,
		0x42, 0x47, 0x42, 0x4f, 0x6d, 0x49, 0x4e, 0x47, 0x64, 0x53,
		0x6d, 0x2b, 0x75, 0x33, 0x7a, 0x67, 0x47, 0x59, 0x52, 0x59,
		0x61, 0x35, 0x66, 0x46, 0x65, 0x59, 0x4e, 0x46, 0x63, 0x54,
		0x57, 0x41, 0x35, 0x74, 0x6b, 0x68, 0x66, 0x61, 0x56, 0x33,
		0x6a, 0x61, 0x4d, 0x61, 0x53, 0x64, 0x75, 0x0a, 0x56, 0x73,
		0x57, 0x31, 0x48, 0x2b, 0x66, 0x48, 0x38, 0x75, 0x36, 0x78,
		0x55, 0x51, 0x4c, 0x57, 0x42, 0x7a, 0x6b, 0x48, 0x2b, 0x4f,
		0x4c, 0x4a, 0x6f, 0x72, 0x76, 0x4f, 0x4b, 0x4c, 0x34, 0x5a,
		0x47, 0x72, 0x46, 0x6e, 0x78, 0x53, 0x47, 0x48, 0x43, 0x42,
		0x45, 0x76, 0x77, 0x31, 0x6e, 0x75, 0x33, 0x70, 0x2b, 0x67,
		0x70, 0x38, 0x67, 0x36, 0x79, 0x35, 0x44, 0x53, 0x36, 0x75,
		0x4a, 0x57, 0x0a, 0x5a, 0x7a, 0x33, 0x58, 0x67, 0x4b, 0x7a,
		0x44, 0x72, 0x66, 0x47, 0x65, 0x6e, 0x59, 0x71, 0x55, 0x46,
		0x6f, 0x71, 0x79, 0x6e, 0x38, 0x2f, 0x69, 0x54, 0x30, 0x49,
		0x34, 0x6d, 0x6f, 0x5a, 0x55, 0x38, 0x49, 0x76, 0x66, 0x56,
		0x68, 0x43, 0x52, 0x7a, 0x79, 0x65, 0x49, 0x36, 0x78, 0x33,
		0x66, 0x64, 0x31, 0x61, 0x57, 0x47, 0x79, 0x5a, 0x55, 0x69,
		0x6c, 0x66, 0x61, 0x48, 0x54, 0x4c, 0x63, 0x0a, 0x49, 0x57,
		0x6b, 0x78, 0x5a, 0x65, 0x65, 0x56, 0x41, 0x67, 0x4d, 0x42,
		0x41, 0x41, 0x45, 0x43, 0x67, 0x67, 0x45, 0x41, 0x53, 0x59,
		0x43, 0x2b, 0x35, 0x7a, 0x75, 0x79, 0x65, 0x65, 0x56, 0x35,
		0x4f, 0x64, 0x36, 0x6d, 0x48, 0x72, 0x54, 0x78, 0x2b, 0x71,
		0x65, 0x48, 0x6e, 0x54, 0x6f, 0x78, 0x31, 0x49, 0x58, 0x46,
		0x34, 0x63, 0x64, 0x72, 0x43, 0x67, 0x70, 0x72, 0x6b, 0x68,
		0x7a, 0x64, 0x0a, 0x39, 0x51, 0x4d, 0x70, 0x53, 0x49, 0x30,
		0x68, 0x44, 0x4f, 0x63, 0x65, 0x69, 0x59, 0x75, 0x45, 0x6e,
		0x5a, 0x72, 0x63, 0x64, 0x2f, 0x50, 0x72, 0x55, 0x6f, 0x57,
		0x43, 0x30, 0x64, 0x41, 0x4d, 0x4f, 0x58, 0x50, 0x53, 0x79,
		0x64, 0x70, 0x46, 0x6b, 0x73, 0x64, 0x4b, 0x42, 0x45, 0x77,
		0x34, 0x77, 0x57, 0x66, 0x70, 0x37, 0x57, 0x68, 0x73, 0x32,
		0x30, 0x49, 0x42, 0x79, 0x76, 0x48, 0x76, 0x0a, 0x32, 0x50,
		0x4e, 0x52, 0x4a, 0x71, 0x4f, 0x58, 0x77, 0x34, 0x71, 0x41,
		0x34, 0x78, 0x4b, 0x44, 0x71, 0x6a, 0x70, 0x75, 0x56, 0x6b,
		0x73, 0x56, 0x59, 0x71, 0x6b, 0x38, 0x74, 0x4e, 0x54, 0x6e,
		0x6f, 0x72, 0x69, 0x73, 0x4f, 0x52, 0x72, 0x4a, 0x6d, 0x41,
		0x7a, 0x4a, 0x6b, 0x57, 0x46, 0x46, 0x36, 0x64, 0x71, 0x5a,
		0x62, 0x31, 0x33, 0x50, 0x4a, 0x73, 0x65, 0x63, 0x61, 0x41,
		0x69, 0x6a, 0x0a, 0x61, 0x41, 0x55, 0x31, 0x56, 0x69, 0x43,
		0x67, 0x4e, 0x5a, 0x67, 0x6a, 0x37, 0x4b, 0x68, 0x68, 0x31,
		0x6e, 0x4f, 0x52, 0x36, 0x5a, 0x47, 0x61, 0x2b, 0x52, 0x5a,
		0x6d, 0x78, 0x6e, 0x4d, 0x4f, 0x6e, 0x4e, 0x74, 0x33, 0x45,
		0x52, 0x4d, 0x57, 0x51, 0x65, 0x4d, 0x32, 0x37, 0x55, 0x39,
		0x55, 0x76, 0x52, 0x51, 0x58, 0x55, 0x30, 0x68, 0x36, 0x2b,
		0x45, 0x7a, 0x4f, 0x7a, 0x69, 0x4d, 0x59, 0x0a, 0x59, 0x77,
		0x43, 0x4e, 0x6b, 0x6c, 0x6b, 0x43, 0x44, 0x33, 0x43, 0x79,
		0x35, 0x4f, 0x63, 0x7a, 0x63, 0x68, 0x58, 0x7a, 0x36, 0x57,
		0x59, 0x48, 0x4a, 0x4e, 0x75, 0x57, 0x36, 0x59, 0x47, 0x54,
		0x42, 0x34, 0x6f, 0x2f, 0x64, 0x4f, 0x45, 0x4c, 0x4b, 0x76,
		0x33, 0x4b, 0x48, 0x42, 0x78, 0x2b, 0x4d, 0x65, 0x45, 0x36,
		0x34, 0x63, 0x52, 0x4d, 0x35, 0x66, 0x73, 0x6e, 0x76, 0x34,
		0x34, 0x48, 0x0a, 0x65, 0x38, 0x34, 0x44, 0x66, 0x59, 0x62,
		0x5a, 0x75, 0x2b, 0x68, 0x56, 0x61, 0x4a, 0x6c, 0x6b, 0x6b,
		0x47, 0x4f, 0x64, 0x76, 0x56, 0x72, 0x53, 0x4e, 0x57, 0x34,
		0x69, 0x73, 0x4f, 0x79, 0x34, 0x73, 0x35, 0x49, 0x33, 0x34,
		0x45, 0x34, 0x78, 0x75, 0x51, 0x4b, 0x42, 0x67, 0x51, 0x44,
		0x4f, 0x30, 0x70, 0x61, 0x69, 0x58, 0x49, 0x68, 0x6f, 0x45,
		0x53, 0x38, 0x76, 0x6f, 0x65, 0x61, 0x53, 0x0a, 0x74, 0x64,
		0x4b, 0x72, 0x34, 0x49, 0x56, 0x32, 0x67, 0x5a, 0x72, 0x48,
		0x65, 0x37, 0x33, 0x5a, 0x4e, 0x61, 0x59, 0x51, 0x53, 0x47,
		0x51, 0x66, 0x30, 0x39, 0x77, 0x51, 0x46, 0x4c, 0x49, 0x68,
		0x36, 0x65, 0x76, 0x55, 0x53, 0x45, 0x2b, 0x37, 0x4e, 0x59,
		0x74, 0x76, 0x30, 0x4a, 0x74, 0x4c, 0x37, 0x67, 0x6d, 0x58,
		0x56, 0x59, 0x72, 0x7a, 0x6a, 0x6a, 0x54, 0x35, 0x77, 0x45,
		0x34, 0x58, 0x0a, 0x48, 0x79, 0x43, 0x33, 0x6a, 0x34, 0x4b,
		0x31, 0x63, 0x6a, 0x35, 0x4e, 0x39, 0x44, 0x58, 0x57, 0x39,
		0x6b, 0x58, 0x63, 0x65, 0x62, 0x44, 0x6a, 0x41, 0x62, 0x68,
		0x6e, 0x30, 0x50, 0x62, 0x50, 0x66, 0x31, 0x78, 0x41, 0x59,
		0x38, 0x59, 0x75, 0x6a, 0x77, 0x56, 0x55, 0x4a, 0x73, 0x56,
		0x63, 0x49, 0x6d, 0x33, 0x2f, 0x73, 0x33, 0x4e, 0x4d, 0x6b,
		0x42, 0x75, 0x5a, 0x52, 0x6f, 0x4e, 0x6c, 0x0a, 0x52, 0x30,
		0x4f, 0x6c, 0x2b, 0x59, 0x61, 0x6a, 0x46, 0x47, 0x58, 0x4b,
		0x62, 0x70, 0x39, 0x50, 0x74, 0x32, 0x34, 0x4f, 0x6f, 0x4b,
		0x71, 0x53, 0x65, 0x77, 0x4b, 0x42, 0x67, 0x51, 0x44, 0x74,
		0x33, 0x2b, 0x51, 0x43, 0x79, 0x66, 0x67, 0x38, 0x73, 0x36,
		0x6b, 0x4e, 0x47, 0x55, 0x73, 0x49, 0x44, 0x31, 0x68, 0x2f,
		0x53, 0x48, 0x77, 0x38, 0x5a, 0x6f, 0x61, 0x4f, 0x4b, 0x6c,
		0x4e, 0x32, 0x0a, 0x61, 0x37, 0x76, 0x32, 0x32, 0x41, 0x7a,
		0x77, 0x30, 0x4f, 0x76, 0x49, 0x58, 0x4d, 0x6c, 0x59, 0x43,
		0x77, 0x44, 0x51, 0x71, 0x6f, 0x42, 0x58, 0x4f, 0x58, 0x75,
		0x7a, 0x50, 0x41, 0x36, 0x74, 0x4b, 0x47, 0x47, 0x69, 0x30,
		0x49, 0x66, 0x31, 0x4f, 0x55, 0x67, 0x52, 0x65, 0x5a, 0x76,
		0x63, 0x32, 0x68, 0x41, 0x70, 0x6a, 0x30, 0x47, 0x4b, 0x73,
		0x69, 0x4d, 0x4d, 0x52, 0x38, 0x44, 0x6f, 0x0a, 0x45, 0x56,
		0x2f, 0x30, 0x70, 0x62, 0x6f, 0x4e, 0x35, 0x68, 0x41, 0x4b,
		0x51, 0x35, 0x43, 0x62, 0x4d, 0x53, 0x71, 0x71, 0x73, 0x44,
		0x2f, 0x2f, 0x74, 0x68, 0x77, 0x4f, 0x52, 0x6a, 0x70, 0x54,
		0x46, 0x34, 0x31, 0x37, 0x6b, 0x64, 0x6c, 0x72, 0x42, 0x64,
		0x34, 0x63, 0x4e, 0x67, 0x6f, 0x4a, 0x41, 0x36, 0x51, 0x6e,
		0x6b, 0x43, 0x7a, 0x2f, 0x63, 0x45, 0x4e, 0x61, 0x6a, 0x46,
		0x4b, 0x4d, 0x0a, 0x42, 0x65, 0x33, 0x32, 0x4f, 0x69, 0x55,
		0x5a, 0x4c, 0x77, 0x4b, 0x42, 0x67, 0x41, 0x77, 0x30, 0x47,
		0x74, 0x56, 0x4c, 0x74, 0x6e, 0x54, 0x46, 0x68, 0x73, 0x34,
		0x39, 0x74, 0x46, 0x67, 0x31, 0x79, 0x45, 0x2b, 0x42, 0x2b,
		0x50, 0x34, 0x4d, 0x47, 0x62, 0x44, 0x70, 0x6d, 0x55, 0x4d,
		0x6d, 0x79, 0x6a, 0x79, 0x74, 0x75, 0x4e, 0x4b, 0x7a, 0x38,
		0x6f, 0x6b, 0x30, 0x58, 0x32, 0x70, 0x6d, 0x0a, 0x33, 0x68,
		0x5a, 0x78, 0x69, 0x64, 0x72, 0x58, 0x42, 0x76, 0x6f, 0x37,
		0x43, 0x64, 0x37, 0x4b, 0x46, 0x65, 0x44, 0x4c, 0x47, 0x46,
		0x37, 0x55, 0x70, 0x6b, 0x57, 0x53, 0x56, 0x6b, 0x6c, 0x41,
		0x2b, 0x66, 0x49, 0x36, 0x30, 0x70, 0x43, 0x59, 0x52, 0x42,
		0x72, 0x66, 0x62, 0x51, 0x6e, 0x5a, 0x45, 0x44, 0x67, 0x35,
		0x4b, 0x66, 0x6e, 0x4c, 0x45, 0x54, 0x46, 0x33, 0x41, 0x41,
		0x6b, 0x77, 0x0a, 0x54, 0x36, 0x59, 0x77, 0x2f, 0x50, 0x47,
		0x6b, 0x57, 0x4d, 0x45, 0x6c, 0x5a, 0x49, 0x48, 0x53, 0x4f,
		0x6b, 0x46, 0x62, 0x4f, 0x6c, 0x4e, 0x72, 0x4d, 0x57, 0x57,
		0x79, 0x35, 0x6d, 0x2f, 0x6b, 0x61, 0x35, 0x5a, 0x77, 0x69,
		0x63, 0x51, 0x33, 0x6f, 0x4f, 0x58, 0x33, 0x37, 0x77, 0x69,
		0x4b, 0x48, 0x30, 0x46, 0x58, 0x73, 0x35, 0x6a, 0x66, 0x41,
		0x6f, 0x47, 0x42, 0x41, 0x49, 0x79, 0x34, 0x0a, 0x63, 0x48,
		0x69, 0x39, 0x35, 0x70, 0x35, 0x79, 0x59, 0x6b, 0x57, 0x66,
		0x65, 0x75, 0x57, 0x76, 0x76, 0x6c, 0x69, 0x2b, 0x46, 0x74,
		0x44, 0x63, 0x6a, 0x69, 0x6b, 0x34, 0x48, 0x4e, 0x76, 0x6e,
		0x61, 0x2f, 0x44, 0x68, 0x61, 0x76, 0x5a, 0x6a, 0x67, 0x62,
		0x69, 0x73, 0x4f, 0x32, 0x68, 0x63, 0x53, 0x67, 0x55, 0x6e,
		0x65, 0x48, 0x6b, 0x68, 0x52, 0x6d, 0x36, 0x48, 0x7a, 0x74,
		0x37, 0x64, 0x0a, 0x44, 0x72, 0x56, 0x30, 0x79, 0x35, 0x6c,
		0x6b, 0x59, 0x36, 0x41, 0x43, 0x34, 0x78, 0x50, 0x4a, 0x35,
		0x4e, 0x6e, 0x52, 0x4a, 0x36, 0x4c, 0x58, 0x4d, 0x79, 0x69,
		0x6d, 0x6f, 0x68, 0x59, 0x6e, 0x67, 0x53, 0x36, 0x48, 0x68,
		0x41, 0x4f, 0x49, 0x34, 0x47, 0x42, 0x53, 0x44, 0x41, 0x2f,
		0x6a, 0x44, 0x43, 0x79, 0x68, 0x79, 0x35, 0x71, 0x49, 0x72,
		0x78, 0x77, 0x36, 0x71, 0x33, 0x4a, 0x51, 0x0a, 0x63, 0x71,
		0x42, 0x78, 0x62, 0x62, 0x2b, 0x45, 0x58, 0x78, 0x73, 0x74,
		0x56, 0x4d, 0x35, 0x30, 0x57, 0x76, 0x34, 0x55, 0x71, 0x71,
		0x6f, 0x4b, 0x54, 0x61, 0x34, 0x5a, 0x61, 0x52, 0x50, 0x77,
		0x78, 0x67, 0x39, 0x74, 0x34, 0x6c, 0x68, 0x39, 0x41, 0x6f,
		0x47, 0x41, 0x45, 0x4f, 0x4f, 0x5a, 0x34, 0x73, 0x30, 0x68,
		0x4f, 0x6a, 0x6f, 0x4d, 0x4c, 0x79, 0x5a, 0x78, 0x36, 0x4e,
		0x58, 0x51, 0x0a, 0x62, 0x69, 0x48, 0x4a, 0x53, 0x32, 0x44,
		0x4e, 0x78, 0x6a, 0x45, 0x34, 0x6f, 0x70, 0x7a, 0x4c, 0x76,
		0x50, 0x36, 0x52, 0x46, 0x6d, 0x71, 0x42, 0x74, 0x67, 0x70,
		0x35, 0x6e, 0x33, 0x77, 0x76, 0x71, 0x76, 0x42, 0x42, 0x4c,
		0x51, 0x4b, 0x4c, 0x67, 0x75, 0x4c, 0x74, 0x69, 0x2b, 0x47,
		0x2f, 0x55, 0x6a, 0x69, 0x79, 0x77, 0x54, 0x46, 0x6c, 0x38,
		0x57, 0x6a, 0x34, 0x50, 0x45, 0x68, 0x6a, 0x0a, 0x54, 0x39,
		0x55, 0x4f, 0x62, 0x55, 0x38, 0x50, 0x75, 0x58, 0x46, 0x33,
		0x58, 0x7a, 0x57, 0x78, 0x33, 0x57, 0x6b, 0x7a, 0x6e, 0x59,
		0x72, 0x46, 0x4c, 0x35, 0x39, 0x4e, 0x31, 0x51, 0x30, 0x31,
		0x2b, 0x64, 0x69, 0x33, 0x31, 0x66, 0x58, 0x4c, 0x46, 0x41,
		0x69, 0x30, 0x34, 0x43, 0x39, 0x4c, 0x49, 0x74, 0x78, 0x48,
		0x72, 0x52, 0x73, 0x6d, 0x59, 0x6d, 0x67, 0x61, 0x50, 0x7a,
		0x6f, 0x4a, 0x0a, 0x6e, 0x5a, 0x50, 0x63, 0x76, 0x79, 0x4c,
		0x6e, 0x70, 0x6c, 0x76, 0x46, 0x4c, 0x50, 0x6f, 0x54, 0x31,
		0x45, 0x42, 0x33, 0x32, 0x66, 0x4d, 0x3d, 0x0a, 0x2d, 0x2d,
		0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x50, 0x52, 0x49,
		0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d,
		0x2d, 0x2d, 0x2d, 0x0a,
	}},
};
/*

RSASHA512 (4096b)

rsa4096-512.	DNSKEY	  256  3  10  AwEAAbt1hC5yoHtsPOVcdrwD4HPEDi1vvEZVxIC5dzdXFM3SdNu0A2rqcHCLK74bEn2hlHWjLAo02dmbLrvA4vS4TNafQwEWDuYvY1VbEGoHV3Q1BGJ1wicRyljq0oh2VecM89nFVVUsVO+vvMz1DB+5KC27RPqGMnoExEzBpEZp1EaBm2nEBnNQ+e8hi0SFtO26rpCpbylRFL76bJjxzVOXSGC9iw8opWldpcAr7UqOvmQoDOBS35OxZfqavJNrZZp7mGHuhZ2dJJQlWSeuZXVitNoRZyBwIXd9brZUs9m/e1fNHKraZOgnI6k/Rb6bmX0BYkWze42TlO4Gxc8BG/sORVN6buD/jkqpVrHU4XLW/MLw8Ir5CHs0CFFTldMdLdSyz40xOySXljIeg8DjaEBSjFKqx+QrIzBHoynTdjaqvP8qLSb+l0gII3BcWYpp7hCGeIc2YjP+ueX32ZgnnEETHL2hGdQDaQCUEcbROIyLlfeagk43sfe73GUxw9eSKjh2XQKAaz6c3rwqM1VlxzRnzamSFzTg4Lz4lvp3Mc9hdTkxgHEhofrW93ujCn8zlzus3Ai6IaF5nO9XaXzhdf9ugySi+U6aWodQM1/8TwIKgiQl8Kw97ZwEqcTZxVQWuGIhYE1hg12M0NIIwWUQ3yIb0OxE1P4ZtN7+ImbM4a4r4oEB
rsa4096-512.	DS    	53381  10  1  BE2E22966D4CE4D059EC75EC37C22F82C8C9F96E
rsa4096-512.	DS    	53381  10  2  177883F8E750941EF032EFEB0B8D1885D438EFD27B3CA155CD411B8E74B1A691
rsa4096-512.	DS    	53381  10  4  81147A3A32A18E3A7A8A0E5E45A1BEEE11F6B09970465C64A883E226EF51265D8EB3308D6CCE6D9098EA5960F3AE64CB

-----BEGIN PRIVATE KEY-----
MIIJQwIBADANBgkqhkiG9w0BAQEFAASCCS0wggkpAgEAAoICAQC7dYQucqB7bDzl
XHa8A+BzxA4tb7xGVcSAuXc3VxTN0nTbtANq6nBwiyu+GxJ9oZR1oywKNNnZmy67
wOL0uEzWn0MBFg7mL2NVWxBqB1d0NQRidcInEcpY6tKIdlXnDPPZxVVVLFTvr7zM
9QwfuSgtu0T6hjJ6BMRMwaRGadRGgZtpxAZzUPnvIYtEhbTtuq6QqW8pURS++myY
8c1Tl0hgvYsPKKVpXaXAK+1Kjr5kKAzgUt+TsWX6mryTa2Wae5hh7oWdnSSUJVkn
rmV1YrTaEWcgcCF3fW62VLPZv3tXzRyq2mToJyOpP0W+m5l9AWJFs3uNk5TuBsXP
ARv7DkVTem7g/45KqVax1OFy1vzC8PCK+Qh7NAhRU5XTHS3Uss+NMTskl5YyHoPA
42hAUoxSqsfkKyMwR6Mp03Y2qrz/Ki0m/pdICCNwXFmKae4QhniHNmIz/rnl99mY
J5xBExy9oRnUA2kAlBHG0TiMi5X3moJON7H3u9xlMcPXkio4dl0CgGs+nN68KjNV
Zcc0Z82pkhc04OC8+Jb6dzHPYXU5MYBxIaH61vd7owp/M5c7rNwIuiGheZzvV2l8
4XX/boMkovlOmlqHUDNf/E8CCoIkJfCsPe2cBKnE2cVUFrhiIWBNYYNdjNDSCMFl
EN8iG9DsRNT+GbTe/iJmzOGuK+KBAQIDAQABAoICAGIJKWGpLOifSJGi862MUXv5
kniMsOgPvr6bQA+/CPBDGj40PH1Tsg4JmozHV7pAIIYNGeEUlsHI+H3UeF0HbZfo
noAY6x3Stp5TC5iH0mLBv1NGxozsLuillgkr6ez1xymim+f5UUownAqMNRT6Hlsc
Twy91fQePhLuSl1Qx2VALuj3TXBNbwhygIULb/3L3bNmCO6Qqsrd6KFsA7etrHp7
Zhg9Z5RdF2wTDPwq/sMedu1yKd7xraWL+wcV6YMLFDxTI6m0HwcjKScpe4ISZwaM
h2RdgAMfpCDS12QPUhAvAskPgiovHeVLY8w1T+YiLSTKotj9z6tT+2sw51M6uWHz
N2pmCCIUzB91PGybT9QELOKOHGhmfXemNUZcro9SotBTLvpOnmM7M8e+qlx6MjkM
2S/Q8PVJQlAibKPZ80YTHxTWsbW/Pm2ZdSWfx50L0sckAsJtlx+4LZoNne+C9i86
X6oLyu/d/Yg+W8N+FsKzmTWo9N50ARly8JhLkUK0rF00pENqrkhKypieo6dbRgRP
cg2rzHpfwVv9nbGy1H2WH3wQXSfkJBWAGr6PRgsCIAQYmRDUO7Zm+KlxUtiVJAZy
K4rltDT7KIDPkdrXn3wBM4S/9WwBL0IXSuzyJagXr17nJPBAuPA+A1mNYZ2Jwc4s
f6ue6a0xbXcN7YaZmNlhAoIBAQDMIjau71M8KtnOB1X7Z0YsEcJ+WwmU+tfQ9uLG
c2ByIxUs0HT7oVykzDPJzy8KbpJXAES+1umLPq9tunMjVrJLTAubVlYFiaJU1MCO
aoui5snIVH9h/wCmumE75+7MtDe5FHwweNfTrV7Jx5Af0H6ZPFPRGubBldhjBa0O
/dG4II9Slqo5J9OIunLlgE4vkHo/LM3NGIAYweCMDyJpupqFJ7YW8l3dbOMQGY8P
CgAlUo88VTBK7D1TE+flQnF3Xz99IhwfUe8LNv8NrCzBNMX2O/LItyqjJ4FD0kwx
0RxGlUSNZA7Te6WpHXMkg2/kudeusmrHgunWRoC6FJXoQIz1AoIBAQDrFrXldq08
N2tTFW83H5km617qsZP1Q2qTAFzxCRcK9D2nHdELYILiR0vzPDdx+8HMVMk4RsrO
FSUEysov4cBXf6JMZM8qpkHVPss5kmRArpBeo0fjPio7rOdk8d/PXbxxQO3qHX/4
WyhgkGNCHwH83f9IzIU8qlPznE5sVfznn1+HCCG63hmTdAySzG8bzQe54jFvOCAZ
5m6Fw/xr2+ZIlGiF38mpMA6KmotLEsuFEIQymtInLBEzFWsVPG2+R6z+JjarQvRD
3ov79MhHYz2V50L0hEmQv+ub8m536HdV70VPfC8tL7NV1DMMpXhg1UMeD6H01c03
jeqeh+0GIpxdAoIBAA5OY4i1Cx9PXX5b80GwRjG2eOwEE1LWlaN2egQwd9Tn8ob2
gQxVjbma+hItq/1OLIFeanL5etqgZQkKpSEX5Vzzkmirj1HKCI9rbadNfkXHaOLa
USccCP6mV0QsiAbepn0ZG28GORIDt33qt4YTRZwVp0YjXYK5rrWZ4zwGu9jspz7B
gmKJEQEnGABPkqsws/CTgkTab7fMdqE6Ok4wnr2wP8mx6LQo+mzZbbY/HC2BcLpo
uf62OcIFxh3CH1OfYEKzzka/pkIh85QQuf48ZZdRMj2w4JMYpK2NInvnTNUBhxFl
rshRhRWqE7S+ZWwhPiVnKaab3SmAPWtNIAhr5bUCggEBAOr1ptTxeREPHrBKYv6I
EUzpv9L3qrZeK/1CRqINgL9M/vWxAkbJrea0dX9kdtHwWnB7aSVa88VnMrmIaO3U
RqXtBewucTPiOTgFa55d5t8S2W4M6YOAlTtRg2/LovIhcH8M8vyjgUxa548joGDs
ltMBme1Lo/rPDdsUJPl4M63YBuTxxqx64LsUKKDrvDzf/dHV9ayAPu1D9sCw8e9n
N9Dgq1rqISZ8xnvXHFLrNQgISug8/r9x9rrEtGLkI5cjEg5taQRghbvziJtIuodG
872IBs5WDSSj/Lq/YZNsfyEfZHmd15Tk8Otei6sfEqJNY6h+3VLUASh+KblKhyYz
n20CggEBAKKtkrmOIkt1IvW2jGsh85bS3ytifN36P+49CnGmLJyJW1Zbylh1Bupl
+mgwLpjIULKe+UTx3UFQBESsF58lTu5NqNl36IUkyG7JOUvTAma/kIAem+XyypkY
04zKWw54dd37KpJtkIgmVEqQI7xztEtMUUTobX1TZjg1/LCAO+SZDYJsU/z93G6z
8a1FnHTFn7DIZ7XuG+7eHAP+8ZEqbrHI7/aoPgXAiogru+fPsDKNNih8n041GmkX
TELR+T1zQl6cYthoXrJzjhx6sn4x3uJt/cfs41crZOY5ALi1m20vZqYlIEWPAw4K
wdEDGd7pmaLwBrX+IUiDuzF04A4j1Pg=
-----END PRIVATE KEY-----

*/
static const key_parameters_t SAMPLE_RSA4096_SHA512_KEY = {
	.name = (uint8_t *)"\x0b""rsa4096-512",
	.flags = 256,
	.protocol = 3,
	.algorithm = 10,
	.public_key = { .size = 516, .data = (uint8_t []){
		0x03, 0x01, 0x00, 0x01, 0xbb, 0x75, 0x84, 0x2e, 0x72, 0xa0,
		0x7b, 0x6c, 0x3c, 0xe5, 0x5c, 0x76, 0xbc, 0x03, 0xe0, 0x73,
		0xc4, 0x0e, 0x2d, 0x6f, 0xbc, 0x46, 0x55, 0xc4, 0x80, 0xb9,
		0x77, 0x37, 0x57, 0x14, 0xcd, 0xd2, 0x74, 0xdb, 0xb4, 0x03,
		0x6a, 0xea, 0x70, 0x70, 0x8b, 0x2b, 0xbe, 0x1b, 0x12, 0x7d,
		0xa1, 0x94, 0x75, 0xa3, 0x2c, 0x0a, 0x34, 0xd9, 0xd9, 0x9b,
		0x2e, 0xbb, 0xc0, 0xe2, 0xf4, 0xb8, 0x4c, 0xd6, 0x9f, 0x43,
		0x01, 0x16, 0x0e, 0xe6, 0x2f, 0x63, 0x55, 0x5b, 0x10, 0x6a,
		0x07, 0x57, 0x74, 0x35, 0x04, 0x62, 0x75, 0xc2, 0x27, 0x11,
		0xca, 0x58, 0xea, 0xd2, 0x88, 0x76, 0x55, 0xe7, 0x0c, 0xf3,
		0xd9, 0xc5, 0x55, 0x55, 0x2c, 0x54, 0xef, 0xaf, 0xbc, 0xcc,
		0xf5, 0x0c, 0x1f, 0xb9, 0x28, 0x2d, 0xbb, 0x44, 0xfa, 0x86,
		0x32, 0x7a, 0x04, 0xc4, 0x4c, 0xc1, 0xa4, 0x46, 0x69, 0xd4,
		0x46, 0x81, 0x9b, 0x69, 0xc4, 0x06, 0x73, 0x50, 0xf9, 0xef,
		0x21, 0x8b, 0x44, 0x85, 0xb4, 0xed, 0xba, 0xae, 0x90, 0xa9,
		0x6f, 0x29, 0x51, 0x14, 0xbe, 0xfa, 0x6c, 0x98, 0xf1, 0xcd,
		0x53, 0x97, 0x48, 0x60, 0xbd, 0x8b, 0x0f, 0x28, 0xa5, 0x69,
		0x5d, 0xa5, 0xc0, 0x2b, 0xed, 0x4a, 0x8e, 0xbe, 0x64, 0x28,
		0x0c, 0xe0, 0x52, 0xdf, 0x93, 0xb1, 0x65, 0xfa, 0x9a, 0xbc,
		0x93, 0x6b, 0x65, 0x9a, 0x7b, 0x98, 0x61, 0xee, 0x85, 0x9d,
		0x9d, 0x24, 0x94, 0x25, 0x59, 0x27, 0xae, 0x65, 0x75, 0x62,
		0xb4, 0xda, 0x11, 0x67, 0x20, 0x70, 0x21, 0x77, 0x7d, 0x6e,
		0xb6, 0x54, 0xb3, 0xd9, 0xbf, 0x7b, 0x57, 0xcd, 0x1c, 0xaa,
		0xda, 0x64, 0xe8, 0x27, 0x23, 0xa9, 0x3f, 0x45, 0xbe, 0x9b,
		0x99, 0x7d, 0x01, 0x62, 0x45, 0xb3, 0x7b, 0x8d, 0x93, 0x94,
		0xee, 0x06, 0xc5, 0xcf, 0x01, 0x1b, 0xfb, 0x0e, 0x45, 0x53,
		0x7a, 0x6e, 0xe0, 0xff, 0x8e, 0x4a, 0xa9, 0x56, 0xb1, 0xd4,
		0xe1, 0x72, 0xd6, 0xfc, 0xc2, 0xf0, 0xf0, 0x8a, 0xf9, 0x08,
		0x7b, 0x34, 0x08, 0x51, 0x53, 0x95, 0xd3, 0x1d, 0x2d, 0xd4,
		0xb2, 0xcf, 0x8d, 0x31, 0x3b, 0x24, 0x97, 0x96, 0x32, 0x1e,
		0x83, 0xc0, 0xe3, 0x68, 0x40, 0x52, 0x8c, 0x52, 0xaa, 0xc7,
		0xe4, 0x2b, 0x23, 0x30, 0x47, 0xa3, 0x29, 0xd3, 0x76, 0x36,
		0xaa, 0xbc, 0xff, 0x2a, 0x2d, 0x26, 0xfe, 0x97, 0x48, 0x08,
		0x23, 0x70, 0x5c, 0x59, 0x8a, 0x69, 0xee, 0x10, 0x86, 0x78,
		0x87, 0x36, 0x62, 0x33, 0xfe, 0xb9, 0xe5, 0xf7, 0xd9, 0x98,
		0x27, 0x9c, 0x41, 0x13, 0x1c, 0xbd, 0xa1, 0x19, 0xd4, 0x03,
		0x69, 0x00, 0x94, 0x11, 0xc6, 0xd1, 0x38, 0x8c, 0x8b, 0x95,
		0xf7, 0x9a, 0x82, 0x4e, 0x37, 0xb1, 0xf7, 0xbb, 0xdc, 0x65,
		0x31, 0xc3, 0xd7, 0x92, 0x2a, 0x38, 0x76, 0x5d, 0x02, 0x80,
		0x6b, 0x3e, 0x9c, 0xde, 0xbc, 0x2a, 0x33, 0x55, 0x65, 0xc7,
		0x34, 0x67, 0xcd, 0xa9, 0x92, 0x17, 0x34, 0xe0, 0xe0, 0xbc,
		0xf8, 0x96, 0xfa, 0x77, 0x31, 0xcf, 0x61, 0x75, 0x39, 0x31,
		0x80, 0x71, 0x21, 0xa1, 0xfa, 0xd6, 0xf7, 0x7b, 0xa3, 0x0a,
		0x7f, 0x33, 0x97, 0x3b, 0xac, 0xdc, 0x08, 0xba, 0x21, 0xa1,
		0x79, 0x9c, 0xef, 0x57, 0x69, 0x7c, 0xe1, 0x75, 0xff, 0x6e,
		0x83, 0x24, 0xa2, 0xf9, 0x4e, 0x9a, 0x5a, 0x87, 0x50, 0x33,
		0x5f, 0xfc, 0x4f, 0x02, 0x0a, 0x82, 0x24, 0x25, 0xf0, 0xac,
		0x3d, 0xed, 0x9c, 0x04, 0xa9, 0xc4, 0xd9, 0xc5, 0x54, 0x16,
		0xb8, 0x62, 0x21, 0x60, 0x4d, 0x61, 0x83, 0x5d, 0x8c, 0xd0,
		0xd2, 0x08, 0xc1, 0x65, 0x10, 0xdf, 0x22, 0x1b, 0xd0, 0xec,
		0x44, 0xd4, 0xfe, 0x19, 0xb4, 0xde, 0xfe, 0x22, 0x66, 0xcc,
		0xe1, 0xae, 0x2b, 0xe2, 0x81, 0x01,
	}},
	.rdata = { .size = 520, .data = (uint8_t []){
		01, 0x00, 0x03, 0x0a,
		0x03, 0x01, 0x00, 0x01, 0xbb, 0x75, 0x84, 0x2e, 0x72, 0xa0,
		0x7b, 0x6c, 0x3c, 0xe5, 0x5c, 0x76, 0xbc, 0x03, 0xe0, 0x73,
		0xc4, 0x0e, 0x2d, 0x6f, 0xbc, 0x46, 0x55, 0xc4, 0x80, 0xb9,
		0x77, 0x37, 0x57, 0x14, 0xcd, 0xd2, 0x74, 0xdb, 0xb4, 0x03,
		0x6a, 0xea, 0x70, 0x70, 0x8b, 0x2b, 0xbe, 0x1b, 0x12, 0x7d,
		0xa1, 0x94, 0x75, 0xa3, 0x2c, 0x0a, 0x34, 0xd9, 0xd9, 0x9b,
		0x2e, 0xbb, 0xc0, 0xe2, 0xf4, 0xb8, 0x4c, 0xd6, 0x9f, 0x43,
		0x01, 0x16, 0x0e, 0xe6, 0x2f, 0x63, 0x55, 0x5b, 0x10, 0x6a,
		0x07, 0x57, 0x74, 0x35, 0x04, 0x62, 0x75, 0xc2, 0x27, 0x11,
		0xca, 0x58, 0xea, 0xd2, 0x88, 0x76, 0x55, 0xe7, 0x0c, 0xf3,
		0xd9, 0xc5, 0x55, 0x55, 0x2c, 0x54, 0xef, 0xaf, 0xbc, 0xcc,
		0xf5, 0x0c, 0x1f, 0xb9, 0x28, 0x2d, 0xbb, 0x44, 0xfa, 0x86,
		0x32, 0x7a, 0x04, 0xc4, 0x4c, 0xc1, 0xa4, 0x46, 0x69, 0xd4,
		0x46, 0x81, 0x9b, 0x69, 0xc4, 0x06, 0x73, 0x50, 0xf9, 0xef,
		0x21, 0x8b, 0x44, 0x85, 0xb4, 0xed, 0xba, 0xae, 0x90, 0xa9,
		0x6f, 0x29, 0x51, 0x14, 0xbe, 0xfa, 0x6c, 0x98, 0xf1, 0xcd,
		0x53, 0x97, 0x48, 0x60, 0xbd, 0x8b, 0x0f, 0x28, 0xa5, 0x69,
		0x5d, 0xa5, 0xc0, 0x2b, 0xed, 0x4a, 0x8e, 0xbe, 0x64, 0x28,
		0x0c, 0xe0, 0x52, 0xdf, 0x93, 0xb1, 0x65, 0xfa, 0x9a, 0xbc,
		0x93, 0x6b, 0x65, 0x9a, 0x7b, 0x98, 0x61, 0xee, 0x85, 0x9d,
		0x9d, 0x24, 0x94, 0x25, 0x59, 0x27, 0xae, 0x65, 0x75, 0x62,
		0xb4, 0xda, 0x11, 0x67, 0x20, 0x70, 0x21, 0x77, 0x7d, 0x6e,
		0xb6, 0x54, 0xb3, 0xd9, 0xbf, 0x7b, 0x57, 0xcd, 0x1c, 0xaa,
		0xda, 0x64, 0xe8, 0x27, 0x23, 0xa9, 0x3f, 0x45, 0xbe, 0x9b,
		0x99, 0x7d, 0x01, 0x62, 0x45, 0xb3, 0x7b, 0x8d, 0x93, 0x94,
		0xee, 0x06, 0xc5, 0xcf, 0x01, 0x1b, 0xfb, 0x0e, 0x45, 0x53,
		0x7a, 0x6e, 0xe0, 0xff, 0x8e, 0x4a, 0xa9, 0x56, 0xb1, 0xd4,
		0xe1, 0x72, 0xd6, 0xfc, 0xc2, 0xf0, 0xf0, 0x8a, 0xf9, 0x08,
		0x7b, 0x34, 0x08, 0x51, 0x53, 0x95, 0xd3, 0x1d, 0x2d, 0xd4,
		0xb2, 0xcf, 0x8d, 0x31, 0x3b, 0x24, 0x97, 0x96, 0x32, 0x1e,
		0x83, 0xc0, 0xe3, 0x68, 0x40, 0x52, 0x8c, 0x52, 0xaa, 0xc7,
		0xe4, 0x2b, 0x23, 0x30, 0x47, 0xa3, 0x29, 0xd3, 0x76, 0x36,
		0xaa, 0xbc, 0xff, 0x2a, 0x2d, 0x26, 0xfe, 0x97, 0x48, 0x08,
		0x23, 0x70, 0x5c, 0x59, 0x8a, 0x69, 0xee, 0x10, 0x86, 0x78,
		0x87, 0x36, 0x62, 0x33, 0xfe, 0xb9, 0xe5, 0xf7, 0xd9, 0x98,
		0x27, 0x9c, 0x41, 0x13, 0x1c, 0xbd, 0xa1, 0x19, 0xd4, 0x03,
		0x69, 0x00, 0x94, 0x11, 0xc6, 0xd1, 0x38, 0x8c, 0x8b, 0x95,
		0xf7, 0x9a, 0x82, 0x4e, 0x37, 0xb1, 0xf7, 0xbb, 0xdc, 0x65,
		0x31, 0xc3, 0xd7, 0x92, 0x2a, 0x38, 0x76, 0x5d, 0x02, 0x80,
		0x6b, 0x3e, 0x9c, 0xde, 0xbc, 0x2a, 0x33, 0x55, 0x65, 0xc7,
		0x34, 0x67, 0xcd, 0xa9, 0x92, 0x17, 0x34, 0xe0, 0xe0, 0xbc,
		0xf8, 0x96, 0xfa, 0x77, 0x31, 0xcf, 0x61, 0x75, 0x39, 0x31,
		0x80, 0x71, 0x21, 0xa1, 0xfa, 0xd6, 0xf7, 0x7b, 0xa3, 0x0a,
		0x7f, 0x33, 0x97, 0x3b, 0xac, 0xdc, 0x08, 0xba, 0x21, 0xa1,
		0x79, 0x9c, 0xef, 0x57, 0x69, 0x7c, 0xe1, 0x75, 0xff, 0x6e,
		0x83, 0x24, 0xa2, 0xf9, 0x4e, 0x9a, 0x5a, 0x87, 0x50, 0x33,
		0x5f, 0xfc, 0x4f, 0x02, 0x0a, 0x82, 0x24, 0x25, 0xf0, 0xac,
		0x3d, 0xed, 0x9c, 0x04, 0xa9, 0xc4, 0xd9, 0xc5, 0x54, 0x16,
		0xb8, 0x62, 0x21, 0x60, 0x4d, 0x61, 0x83, 0x5d, 0x8c, 0xd0,
		0xd2, 0x08, 0xc1, 0x65, 0x10, 0xdf, 0x22, 0x1b, 0xd0, 0xec,
		0x44, 0xd4, 0xfe, 0x19, 0xb4, 0xde, 0xfe, 0x22, 0x66, 0xcc,
		0xe1, 0xae, 0x2b, 0xe2, 0x81, 0x01,
	}},
	.key_id = "48d43d5cd5b5b3cd5d374a04573cb908003b6005",
	.keytag = 53381,
	.ds_sha1 = { .size = 24, .data = (uint8_t []){
		0xd0, 0x85, 0x0a, 0x01, 0xbe, 0x2e, 0x22, 0x96, 0x6d, 0x4c,
		0xe4, 0xd0, 0x59, 0xec, 0x75, 0xec, 0x37, 0xc2, 0x2f, 0x82,
		0xc8, 0xc9, 0xf9, 0x6e,
	}},
	.ds_sha256 = { .size = 36, .data = (uint8_t []){
		0xd0, 0x85, 0x0a, 0x02, 0x17, 0x78, 0x83, 0xf8, 0xe7, 0x50,
		0x94, 0x1e, 0xf0, 0x32, 0xef, 0xeb, 0x0b, 0x8d, 0x18, 0x85,
		0xd4, 0x38, 0xef, 0xd2, 0x7b, 0x3c, 0xa1, 0x55, 0xcd, 0x41,
		0x1b, 0x8e, 0x74, 0xb1, 0xa6, 0x91,
	}},
	.ds_sha384 = { .size = 52, .data = (uint8_t []){
		0xd0, 0x85, 0x0a, 0x04, 0x81, 0x14, 0x7a, 0x3a, 0x32, 0xa1,
		0x8e, 0x3a, 0x7a, 0x8a, 0x0e, 0x5e, 0x45, 0xa1, 0xbe, 0xee,
		0x11, 0xf6, 0xb0, 0x99, 0x70, 0x46, 0x5c, 0x64, 0xa8, 0x83,
		0xe2, 0x26, 0xef, 0x51, 0x26, 0x5d, 0x8e, 0xb3, 0x30, 0x8d,
		0x6c, 0xce, 0x6d, 0x90, 0x98, 0xea, 0x59, 0x60, 0xf3, 0xae,
		0x64, 0xcb,
	}},
	.bit_size = 4096,
	.pem = { .size = 3272, .data = (uint8_t []){
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49,
		0x49, 0x4a, 0x51, 0x77, 0x49, 0x42, 0x41, 0x44, 0x41, 0x4e,
		0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77,
		0x30, 0x42, 0x41, 0x51, 0x45, 0x46, 0x41, 0x41, 0x53, 0x43,
		0x43, 0x53, 0x30, 0x77, 0x67, 0x67, 0x6b, 0x70, 0x41, 0x67,
		0x45, 0x41, 0x41, 0x6f, 0x49, 0x43, 0x41, 0x51, 0x43, 0x37,
		0x64, 0x59, 0x51, 0x75, 0x63, 0x71, 0x42, 0x37, 0x62, 0x44,
		0x7a, 0x6c, 0x0a, 0x58, 0x48, 0x61, 0x38, 0x41, 0x2b, 0x42,
		0x7a, 0x78, 0x41, 0x34, 0x74, 0x62, 0x37, 0x78, 0x47, 0x56,
		0x63, 0x53, 0x41, 0x75, 0x58, 0x63, 0x33, 0x56, 0x78, 0x54,
		0x4e, 0x30, 0x6e, 0x54, 0x62, 0x74, 0x41, 0x4e, 0x71, 0x36,
		0x6e, 0x42, 0x77, 0x69, 0x79, 0x75, 0x2b, 0x47, 0x78, 0x4a,
		0x39, 0x6f, 0x5a, 0x52, 0x31, 0x6f, 0x79, 0x77, 0x4b, 0x4e,
		0x4e, 0x6e, 0x5a, 0x6d, 0x79, 0x36, 0x37, 0x0a, 0x77, 0x4f,
		0x4c, 0x30, 0x75, 0x45, 0x7a, 0x57, 0x6e, 0x30, 0x4d, 0x42,
		0x46, 0x67, 0x37, 0x6d, 0x4c, 0x32, 0x4e, 0x56, 0x57, 0x78,
		0x42, 0x71, 0x42, 0x31, 0x64, 0x30, 0x4e, 0x51, 0x52, 0x69,
		0x64, 0x63, 0x49, 0x6e, 0x45, 0x63, 0x70, 0x59, 0x36, 0x74,
		0x4b, 0x49, 0x64, 0x6c, 0x58, 0x6e, 0x44, 0x50, 0x50, 0x5a,
		0x78, 0x56, 0x56, 0x56, 0x4c, 0x46, 0x54, 0x76, 0x72, 0x37,
		0x7a, 0x4d, 0x0a, 0x39, 0x51, 0x77, 0x66, 0x75, 0x53, 0x67,
		0x74, 0x75, 0x30, 0x54, 0x36, 0x68, 0x6a, 0x4a, 0x36, 0x42,
		0x4d, 0x52, 0x4d, 0x77, 0x61, 0x52, 0x47, 0x61, 0x64, 0x52,
		0x47, 0x67, 0x5a, 0x74, 0x70, 0x78, 0x41, 0x5a, 0x7a, 0x55,
		0x50, 0x6e, 0x76, 0x49, 0x59, 0x74, 0x45, 0x68, 0x62, 0x54,
		0x74, 0x75, 0x71, 0x36, 0x51, 0x71, 0x57, 0x38, 0x70, 0x55,
		0x52, 0x53, 0x2b, 0x2b, 0x6d, 0x79, 0x59, 0x0a, 0x38, 0x63,
		0x31, 0x54, 0x6c, 0x30, 0x68, 0x67, 0x76, 0x59, 0x73, 0x50,
		0x4b, 0x4b, 0x56, 0x70, 0x58, 0x61, 0x58, 0x41, 0x4b, 0x2b,
		0x31, 0x4b, 0x6a, 0x72, 0x35, 0x6b, 0x4b, 0x41, 0x7a, 0x67,
		0x55, 0x74, 0x2b, 0x54, 0x73, 0x57, 0x58, 0x36, 0x6d, 0x72,
		0x79, 0x54, 0x61, 0x32, 0x57, 0x61, 0x65, 0x35, 0x68, 0x68,
		0x37, 0x6f, 0x57, 0x64, 0x6e, 0x53, 0x53, 0x55, 0x4a, 0x56,
		0x6b, 0x6e, 0x0a, 0x72, 0x6d, 0x56, 0x31, 0x59, 0x72, 0x54,
		0x61, 0x45, 0x57, 0x63, 0x67, 0x63, 0x43, 0x46, 0x33, 0x66,
		0x57, 0x36, 0x32, 0x56, 0x4c, 0x50, 0x5a, 0x76, 0x33, 0x74,
		0x58, 0x7a, 0x52, 0x79, 0x71, 0x32, 0x6d, 0x54, 0x6f, 0x4a,
		0x79, 0x4f, 0x70, 0x50, 0x30, 0x57, 0x2b, 0x6d, 0x35, 0x6c,
		0x39, 0x41, 0x57, 0x4a, 0x46, 0x73, 0x33, 0x75, 0x4e, 0x6b,
		0x35, 0x54, 0x75, 0x42, 0x73, 0x58, 0x50, 0x0a, 0x41, 0x52,
		0x76, 0x37, 0x44, 0x6b, 0x56, 0x54, 0x65, 0x6d, 0x37, 0x67,
		0x2f, 0x34, 0x35, 0x4b, 0x71, 0x56, 0x61, 0x78, 0x31, 0x4f,
		0x46, 0x79, 0x31, 0x76, 0x7a, 0x43, 0x38, 0x50, 0x43, 0x4b,
		0x2b, 0x51, 0x68, 0x37, 0x4e, 0x41, 0x68, 0x52, 0x55, 0x35,
		0x58, 0x54, 0x48, 0x53, 0x33, 0x55, 0x73, 0x73, 0x2b, 0x4e,
		0x4d, 0x54, 0x73, 0x6b, 0x6c, 0x35, 0x59, 0x79, 0x48, 0x6f,
		0x50, 0x41, 0x0a, 0x34, 0x32, 0x68, 0x41, 0x55, 0x6f, 0x78,
		0x53, 0x71, 0x73, 0x66, 0x6b, 0x4b, 0x79, 0x4d, 0x77, 0x52,
		0x36, 0x4d, 0x70, 0x30, 0x33, 0x59, 0x32, 0x71, 0x72, 0x7a,
		0x2f, 0x4b, 0x69, 0x30, 0x6d, 0x2f, 0x70, 0x64, 0x49, 0x43,
		0x43, 0x4e, 0x77, 0x58, 0x46, 0x6d, 0x4b, 0x61, 0x65, 0x34,
		0x51, 0x68, 0x6e, 0x69, 0x48, 0x4e, 0x6d, 0x49, 0x7a, 0x2f,
		0x72, 0x6e, 0x6c, 0x39, 0x39, 0x6d, 0x59, 0x0a, 0x4a, 0x35,
		0x78, 0x42, 0x45, 0x78, 0x79, 0x39, 0x6f, 0x52, 0x6e, 0x55,
		0x41, 0x32, 0x6b, 0x41, 0x6c, 0x42, 0x48, 0x47, 0x30, 0x54,
		0x69, 0x4d, 0x69, 0x35, 0x58, 0x33, 0x6d, 0x6f, 0x4a, 0x4f,
		0x4e, 0x37, 0x48, 0x33, 0x75, 0x39, 0x78, 0x6c, 0x4d, 0x63,
		0x50, 0x58, 0x6b, 0x69, 0x6f, 0x34, 0x64, 0x6c, 0x30, 0x43,
		0x67, 0x47, 0x73, 0x2b, 0x6e, 0x4e, 0x36, 0x38, 0x4b, 0x6a,
		0x4e, 0x56, 0x0a, 0x5a, 0x63, 0x63, 0x30, 0x5a, 0x38, 0x32,
		0x70, 0x6b, 0x68, 0x63, 0x30, 0x34, 0x4f, 0x43, 0x38, 0x2b,
		0x4a, 0x62, 0x36, 0x64, 0x7a, 0x48, 0x50, 0x59, 0x58, 0x55,
		0x35, 0x4d, 0x59, 0x42, 0x78, 0x49, 0x61, 0x48, 0x36, 0x31,
		0x76, 0x64, 0x37, 0x6f, 0x77, 0x70, 0x2f, 0x4d, 0x35, 0x63,
		0x37, 0x72, 0x4e, 0x77, 0x49, 0x75, 0x69, 0x47, 0x68, 0x65,
		0x5a, 0x7a, 0x76, 0x56, 0x32, 0x6c, 0x38, 0x0a, 0x34, 0x58,
		0x58, 0x2f, 0x62, 0x6f, 0x4d, 0x6b, 0x6f, 0x76, 0x6c, 0x4f,
		0x6d, 0x6c, 0x71, 0x48, 0x55, 0x44, 0x4e, 0x66, 0x2f, 0x45,
		0x38, 0x43, 0x43, 0x6f, 0x49, 0x6b, 0x4a, 0x66, 0x43, 0x73,
		0x50, 0x65, 0x32, 0x63, 0x42, 0x4b, 0x6e, 0x45, 0x32, 0x63,
		0x56, 0x55, 0x46, 0x72, 0x68, 0x69, 0x49, 0x57, 0x42, 0x4e,
		0x59, 0x59, 0x4e, 0x64, 0x6a, 0x4e, 0x44, 0x53, 0x43, 0x4d,
		0x46, 0x6c, 0x0a, 0x45, 0x4e, 0x38, 0x69, 0x47, 0x39, 0x44,
		0x73, 0x52, 0x4e, 0x54, 0x2b, 0x47, 0x62, 0x54, 0x65, 0x2f,
		0x69, 0x4a, 0x6d, 0x7a, 0x4f, 0x47, 0x75, 0x4b, 0x2b, 0x4b,
		0x42, 0x41, 0x51, 0x49, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41,
		0x6f, 0x49, 0x43, 0x41, 0x47, 0x49, 0x4a, 0x4b, 0x57, 0x47,
		0x70, 0x4c, 0x4f, 0x69, 0x66, 0x53, 0x4a, 0x47, 0x69, 0x38,
		0x36, 0x32, 0x4d, 0x55, 0x58, 0x76, 0x35, 0x0a, 0x6b, 0x6e,
		0x69, 0x4d, 0x73, 0x4f, 0x67, 0x50, 0x76, 0x72, 0x36, 0x62,
		0x51, 0x41, 0x2b, 0x2f, 0x43, 0x50, 0x42, 0x44, 0x47, 0x6a,
		0x34, 0x30, 0x50, 0x48, 0x31, 0x54, 0x73, 0x67, 0x34, 0x4a,
		0x6d, 0x6f, 0x7a, 0x48, 0x56, 0x37, 0x70, 0x41, 0x49, 0x49,
		0x59, 0x4e, 0x47, 0x65, 0x45, 0x55, 0x6c, 0x73, 0x48, 0x49,
		0x2b, 0x48, 0x33, 0x55, 0x65, 0x46, 0x30, 0x48, 0x62, 0x5a,
		0x66, 0x6f, 0x0a, 0x6e, 0x6f, 0x41, 0x59, 0x36, 0x78, 0x33,
		0x53, 0x74, 0x70, 0x35, 0x54, 0x43, 0x35, 0x69, 0x48, 0x30,
		0x6d, 0x4c, 0x42, 0x76, 0x31, 0x4e, 0x47, 0x78, 0x6f, 0x7a,
		0x73, 0x4c, 0x75, 0x69, 0x6c, 0x6c, 0x67, 0x6b, 0x72, 0x36,
		0x65, 0x7a, 0x31, 0x78, 0x79, 0x6d, 0x69, 0x6d, 0x2b, 0x66,
		0x35, 0x55, 0x55, 0x6f, 0x77, 0x6e, 0x41, 0x71, 0x4d, 0x4e,
		0x52, 0x54, 0x36, 0x48, 0x6c, 0x73, 0x63, 0x0a, 0x54, 0x77,
		0x79, 0x39, 0x31, 0x66, 0x51, 0x65, 0x50, 0x68, 0x4c, 0x75,
		0x53, 0x6c, 0x31, 0x51, 0x78, 0x32, 0x56, 0x41, 0x4c, 0x75,
		0x6a, 0x33, 0x54, 0x58, 0x42, 0x4e, 0x62, 0x77, 0x68, 0x79,
		0x67, 0x49, 0x55, 0x4c, 0x62, 0x2f, 0x33, 0x4c, 0x33, 0x62,
		0x4e, 0x6d, 0x43, 0x4f, 0x36, 0x51, 0x71, 0x73, 0x72, 0x64,
		0x36, 0x4b, 0x46, 0x73, 0x41, 0x37, 0x65, 0x74, 0x72, 0x48,
		0x70, 0x37, 0x0a, 0x5a, 0x68, 0x67, 0x39, 0x5a, 0x35, 0x52,
		0x64, 0x46, 0x32, 0x77, 0x54, 0x44, 0x50, 0x77, 0x71, 0x2f,
		0x73, 0x4d, 0x65, 0x64, 0x75, 0x31, 0x79, 0x4b, 0x64, 0x37,
		0x78, 0x72, 0x61, 0x57, 0x4c, 0x2b, 0x77, 0x63, 0x56, 0x36,
		0x59, 0x4d, 0x4c, 0x46, 0x44, 0x78, 0x54, 0x49, 0x36, 0x6d,
		0x30, 0x48, 0x77, 0x63, 0x6a, 0x4b, 0x53, 0x63, 0x70, 0x65,
		0x34, 0x49, 0x53, 0x5a, 0x77, 0x61, 0x4d, 0x0a, 0x68, 0x32,
		0x52, 0x64, 0x67, 0x41, 0x4d, 0x66, 0x70, 0x43, 0x44, 0x53,
		0x31, 0x32, 0x51, 0x50, 0x55, 0x68, 0x41, 0x76, 0x41, 0x73,
		0x6b, 0x50, 0x67, 0x69, 0x6f, 0x76, 0x48, 0x65, 0x56, 0x4c,
		0x59, 0x38, 0x77, 0x31, 0x54, 0x2b, 0x59, 0x69, 0x4c, 0x53,
		0x54, 0x4b, 0x6f, 0x74, 0x6a, 0x39, 0x7a, 0x36, 0x74, 0x54,
		0x2b, 0x32, 0x73, 0x77, 0x35, 0x31, 0x4d, 0x36, 0x75, 0x57,
		0x48, 0x7a, 0x0a, 0x4e, 0x32, 0x70, 0x6d, 0x43, 0x43, 0x49,
		0x55, 0x7a, 0x42, 0x39, 0x31, 0x50, 0x47, 0x79, 0x62, 0x54,
		0x39, 0x51, 0x45, 0x4c, 0x4f, 0x4b, 0x4f, 0x48, 0x47, 0x68,
		0x6d, 0x66, 0x58, 0x65, 0x6d, 0x4e, 0x55, 0x5a, 0x63, 0x72,
		0x6f, 0x39, 0x53, 0x6f, 0x74, 0x42, 0x54, 0x4c, 0x76, 0x70,
		0x4f, 0x6e, 0x6d, 0x4d, 0x37, 0x4d, 0x38, 0x65, 0x2b, 0x71,
		0x6c, 0x78, 0x36, 0x4d, 0x6a, 0x6b, 0x4d, 0x0a, 0x32, 0x53,
		0x2f, 0x51, 0x38, 0x50, 0x56, 0x4a, 0x51, 0x6c, 0x41, 0x69,
		0x62, 0x4b, 0x50, 0x5a, 0x38, 0x30, 0x59, 0x54, 0x48, 0x78,
		0x54, 0x57, 0x73, 0x62, 0x57, 0x2f, 0x50, 0x6d, 0x32, 0x5a,
		0x64, 0x53, 0x57, 0x66, 0x78, 0x35, 0x30, 0x4c, 0x30, 0x73,
		0x63, 0x6b, 0x41, 0x73, 0x4a, 0x74, 0x6c, 0x78, 0x2b, 0x34,
		0x4c, 0x5a, 0x6f, 0x4e, 0x6e, 0x65, 0x2b, 0x43, 0x39, 0x69,
		0x38, 0x36, 0x0a, 0x58, 0x36, 0x6f, 0x4c, 0x79, 0x75, 0x2f,
		0x64, 0x2f, 0x59, 0x67, 0x2b, 0x57, 0x38, 0x4e, 0x2b, 0x46,
		0x73, 0x4b, 0x7a, 0x6d, 0x54, 0x57, 0x6f, 0x39, 0x4e, 0x35,
		0x30, 0x41, 0x52, 0x6c, 0x79, 0x38, 0x4a, 0x68, 0x4c, 0x6b,
		0x55, 0x4b, 0x30, 0x72, 0x46, 0x30, 0x30, 0x70, 0x45, 0x4e,
		0x71, 0x72, 0x6b, 0x68, 0x4b, 0x79, 0x70, 0x69, 0x65, 0x6f,
		0x36, 0x64, 0x62, 0x52, 0x67, 0x52, 0x50, 0x0a, 0x63, 0x67,
		0x32, 0x72, 0x7a, 0x48, 0x70, 0x66, 0x77, 0x56, 0x76, 0x39,
		0x6e, 0x62, 0x47, 0x79, 0x31, 0x48, 0x32, 0x57, 0x48, 0x33,
		0x77, 0x51, 0x58, 0x53, 0x66, 0x6b, 0x4a, 0x42, 0x57, 0x41,
		0x47, 0x72, 0x36, 0x50, 0x52, 0x67, 0x73, 0x43, 0x49, 0x41,
		0x51, 0x59, 0x6d, 0x52, 0x44, 0x55, 0x4f, 0x37, 0x5a, 0x6d,
		0x2b, 0x4b, 0x6c, 0x78, 0x55, 0x74, 0x69, 0x56, 0x4a, 0x41,
		0x5a, 0x79, 0x0a, 0x4b, 0x34, 0x72, 0x6c, 0x74, 0x44, 0x54,
		0x37, 0x4b, 0x49, 0x44, 0x50, 0x6b, 0x64, 0x72, 0x58, 0x6e,
		0x33, 0x77, 0x42, 0x4d, 0x34, 0x53, 0x2f, 0x39, 0x57, 0x77,
		0x42, 0x4c, 0x30, 0x49, 0x58, 0x53, 0x75, 0x7a, 0x79, 0x4a,
		0x61, 0x67, 0x58, 0x72, 0x31, 0x37, 0x6e, 0x4a, 0x50, 0x42,
		0x41, 0x75, 0x50, 0x41, 0x2b, 0x41, 0x31, 0x6d, 0x4e, 0x59,
		0x5a, 0x32, 0x4a, 0x77, 0x63, 0x34, 0x73, 0x0a, 0x66, 0x36,
		0x75, 0x65, 0x36, 0x61, 0x30, 0x78, 0x62, 0x58, 0x63, 0x4e,
		0x37, 0x59, 0x61, 0x5a, 0x6d, 0x4e, 0x6c, 0x68, 0x41, 0x6f,
		0x49, 0x42, 0x41, 0x51, 0x44, 0x4d, 0x49, 0x6a, 0x61, 0x75,
		0x37, 0x31, 0x4d, 0x38, 0x4b, 0x74, 0x6e, 0x4f, 0x42, 0x31,
		0x58, 0x37, 0x5a, 0x30, 0x59, 0x73, 0x45, 0x63, 0x4a, 0x2b,
		0x57, 0x77, 0x6d, 0x55, 0x2b, 0x74, 0x66, 0x51, 0x39, 0x75,
		0x4c, 0x47, 0x0a, 0x63, 0x32, 0x42, 0x79, 0x49, 0x78, 0x55,
		0x73, 0x30, 0x48, 0x54, 0x37, 0x6f, 0x56, 0x79, 0x6b, 0x7a,
		0x44, 0x50, 0x4a, 0x7a, 0x79, 0x38, 0x4b, 0x62, 0x70, 0x4a,
		0x58, 0x41, 0x45, 0x53, 0x2b, 0x31, 0x75, 0x6d, 0x4c, 0x50,
		0x71, 0x39, 0x74, 0x75, 0x6e, 0x4d, 0x6a, 0x56, 0x72, 0x4a,
		0x4c, 0x54, 0x41, 0x75, 0x62, 0x56, 0x6c, 0x59, 0x46, 0x69,
		0x61, 0x4a, 0x55, 0x31, 0x4d, 0x43, 0x4f, 0x0a, 0x61, 0x6f,
		0x75, 0x69, 0x35, 0x73, 0x6e, 0x49, 0x56, 0x48, 0x39, 0x68,
		0x2f, 0x77, 0x43, 0x6d, 0x75, 0x6d, 0x45, 0x37, 0x35, 0x2b,
		0x37, 0x4d, 0x74, 0x44, 0x65, 0x35, 0x46, 0x48, 0x77, 0x77,
		0x65, 0x4e, 0x66, 0x54, 0x72, 0x56, 0x37, 0x4a, 0x78, 0x35,
		0x41, 0x66, 0x30, 0x48, 0x36, 0x5a, 0x50, 0x46, 0x50, 0x52,
		0x47, 0x75, 0x62, 0x42, 0x6c, 0x64, 0x68, 0x6a, 0x42, 0x61,
		0x30, 0x4f, 0x0a, 0x2f, 0x64, 0x47, 0x34, 0x49, 0x49, 0x39,
		0x53, 0x6c, 0x71, 0x6f, 0x35, 0x4a, 0x39, 0x4f, 0x49, 0x75,
		0x6e, 0x4c, 0x6c, 0x67, 0x45, 0x34, 0x76, 0x6b, 0x48, 0x6f,
		0x2f, 0x4c, 0x4d, 0x33, 0x4e, 0x47, 0x49, 0x41, 0x59, 0x77,
		0x65, 0x43, 0x4d, 0x44, 0x79, 0x4a, 0x70, 0x75, 0x70, 0x71,
		0x46, 0x4a, 0x37, 0x59, 0x57, 0x38, 0x6c, 0x33, 0x64, 0x62,
		0x4f, 0x4d, 0x51, 0x47, 0x59, 0x38, 0x50, 0x0a, 0x43, 0x67,
		0x41, 0x6c, 0x55, 0x6f, 0x38, 0x38, 0x56, 0x54, 0x42, 0x4b,
		0x37, 0x44, 0x31, 0x54, 0x45, 0x2b, 0x66, 0x6c, 0x51, 0x6e,
		0x46, 0x33, 0x58, 0x7a, 0x39, 0x39, 0x49, 0x68, 0x77, 0x66,
		0x55, 0x65, 0x38, 0x4c, 0x4e, 0x76, 0x38, 0x4e, 0x72, 0x43,
		0x7a, 0x42, 0x4e, 0x4d, 0x58, 0x32, 0x4f, 0x2f, 0x4c, 0x49,
		0x74, 0x79, 0x71, 0x6a, 0x4a, 0x34, 0x46, 0x44, 0x30, 0x6b,
		0x77, 0x78, 0x0a, 0x30, 0x52, 0x78, 0x47, 0x6c, 0x55, 0x53,
		0x4e, 0x5a, 0x41, 0x37, 0x54, 0x65, 0x36, 0x57, 0x70, 0x48,
		0x58, 0x4d, 0x6b, 0x67, 0x32, 0x2f, 0x6b, 0x75, 0x64, 0x65,
		0x75, 0x73, 0x6d, 0x72, 0x48, 0x67, 0x75, 0x6e, 0x57, 0x52,
		0x6f, 0x43, 0x36, 0x46, 0x4a, 0x58, 0x6f, 0x51, 0x49, 0x7a,
		0x31, 0x41, 0x6f, 0x49, 0x42, 0x41, 0x51, 0x44, 0x72, 0x46,
		0x72, 0x58, 0x6c, 0x64, 0x71, 0x30, 0x38, 0x0a, 0x4e, 0x32,
		0x74, 0x54, 0x46, 0x57, 0x38, 0x33, 0x48, 0x35, 0x6b, 0x6d,
		0x36, 0x31, 0x37, 0x71, 0x73, 0x5a, 0x50, 0x31, 0x51, 0x32,
		0x71, 0x54, 0x41, 0x46, 0x7a, 0x78, 0x43, 0x52, 0x63, 0x4b,
		0x39, 0x44, 0x32, 0x6e, 0x48, 0x64, 0x45, 0x4c, 0x59, 0x49,
		0x4c, 0x69, 0x52, 0x30, 0x76, 0x7a, 0x50, 0x44, 0x64, 0x78,
		0x2b, 0x38, 0x48, 0x4d, 0x56, 0x4d, 0x6b, 0x34, 0x52, 0x73,
		0x72, 0x4f, 0x0a, 0x46, 0x53, 0x55, 0x45, 0x79, 0x73, 0x6f,
		0x76, 0x34, 0x63, 0x42, 0x58, 0x66, 0x36, 0x4a, 0x4d, 0x5a,
		0x4d, 0x38, 0x71, 0x70, 0x6b, 0x48, 0x56, 0x50, 0x73, 0x73,
		0x35, 0x6b, 0x6d, 0x52, 0x41, 0x72, 0x70, 0x42, 0x65, 0x6f,
		0x30, 0x66, 0x6a, 0x50, 0x69, 0x6f, 0x37, 0x72, 0x4f, 0x64,
		0x6b, 0x38, 0x64, 0x2f, 0x50, 0x58, 0x62, 0x78, 0x78, 0x51,
		0x4f, 0x33, 0x71, 0x48, 0x58, 0x2f, 0x34, 0x0a, 0x57, 0x79,
		0x68, 0x67, 0x6b, 0x47, 0x4e, 0x43, 0x48, 0x77, 0x48, 0x38,
		0x33, 0x66, 0x39, 0x49, 0x7a, 0x49, 0x55, 0x38, 0x71, 0x6c,
		0x50, 0x7a, 0x6e, 0x45, 0x35, 0x73, 0x56, 0x66, 0x7a, 0x6e,
		0x6e, 0x31, 0x2b, 0x48, 0x43, 0x43, 0x47, 0x36, 0x33, 0x68,
		0x6d, 0x54, 0x64, 0x41, 0x79, 0x53, 0x7a, 0x47, 0x38, 0x62,
		0x7a, 0x51, 0x65, 0x35, 0x34, 0x6a, 0x46, 0x76, 0x4f, 0x43,
		0x41, 0x5a, 0x0a, 0x35, 0x6d, 0x36, 0x46, 0x77, 0x2f, 0x78,
		0x72, 0x32, 0x2b, 0x5a, 0x49, 0x6c, 0x47, 0x69, 0x46, 0x33,
		0x38, 0x6d, 0x70, 0x4d, 0x41, 0x36, 0x4b, 0x6d, 0x6f, 0x74,
		0x4c, 0x45, 0x73, 0x75, 0x46, 0x45, 0x49, 0x51, 0x79, 0x6d,
		0x74, 0x49, 0x6e, 0x4c, 0x42, 0x45, 0x7a, 0x46, 0x57, 0x73,
		0x56, 0x50, 0x47, 0x32, 0x2b, 0x52, 0x36, 0x7a, 0x2b, 0x4a,
		0x6a, 0x61, 0x72, 0x51, 0x76, 0x52, 0x44, 0x0a, 0x33, 0x6f,
		0x76, 0x37, 0x39, 0x4d, 0x68, 0x48, 0x59, 0x7a, 0x32, 0x56,
		0x35, 0x30, 0x4c, 0x30, 0x68, 0x45, 0x6d, 0x51, 0x76, 0x2b,
		0x75, 0x62, 0x38, 0x6d, 0x35, 0x33, 0x36, 0x48, 0x64, 0x56,
		0x37, 0x30, 0x56, 0x50, 0x66, 0x43, 0x38, 0x74, 0x4c, 0x37,
		0x4e, 0x56, 0x31, 0x44, 0x4d, 0x4d, 0x70, 0x58, 0x68, 0x67,
		0x31, 0x55, 0x4d, 0x65, 0x44, 0x36, 0x48, 0x30, 0x31, 0x63,
		0x30, 0x33, 0x0a, 0x6a, 0x65, 0x71, 0x65, 0x68, 0x2b, 0x30,
		0x47, 0x49, 0x70, 0x78, 0x64, 0x41, 0x6f, 0x49, 0x42, 0x41,
		0x41, 0x35, 0x4f, 0x59, 0x34, 0x69, 0x31, 0x43, 0x78, 0x39,
		0x50, 0x58, 0x58, 0x35, 0x62, 0x38, 0x30, 0x47, 0x77, 0x52,
		0x6a, 0x47, 0x32, 0x65, 0x4f, 0x77, 0x45, 0x45, 0x31, 0x4c,
		0x57, 0x6c, 0x61, 0x4e, 0x32, 0x65, 0x67, 0x51, 0x77, 0x64,
		0x39, 0x54, 0x6e, 0x38, 0x6f, 0x62, 0x32, 0x0a, 0x67, 0x51,
		0x78, 0x56, 0x6a, 0x62, 0x6d, 0x61, 0x2b, 0x68, 0x49, 0x74,
		0x71, 0x2f, 0x31, 0x4f, 0x4c, 0x49, 0x46, 0x65, 0x61, 0x6e,
		0x4c, 0x35, 0x65, 0x74, 0x71, 0x67, 0x5a, 0x51, 0x6b, 0x4b,
		0x70, 0x53, 0x45, 0x58, 0x35, 0x56, 0x7a, 0x7a, 0x6b, 0x6d,
		0x69, 0x72, 0x6a, 0x31, 0x48, 0x4b, 0x43, 0x49, 0x39, 0x72,
		0x62, 0x61, 0x64, 0x4e, 0x66, 0x6b, 0x58, 0x48, 0x61, 0x4f,
		0x4c, 0x61, 0x0a, 0x55, 0x53, 0x63, 0x63, 0x43, 0x50, 0x36,
		0x6d, 0x56, 0x30, 0x51, 0x73, 0x69, 0x41, 0x62, 0x65, 0x70,
		0x6e, 0x30, 0x5a, 0x47, 0x32, 0x38, 0x47, 0x4f, 0x52, 0x49,
		0x44, 0x74, 0x33, 0x33, 0x71, 0x74, 0x34, 0x59, 0x54, 0x52,
		0x5a, 0x77, 0x56, 0x70, 0x30, 0x59, 0x6a, 0x58, 0x59, 0x4b,
		0x35, 0x72, 0x72, 0x57, 0x5a, 0x34, 0x7a, 0x77, 0x47, 0x75,
		0x39, 0x6a, 0x73, 0x70, 0x7a, 0x37, 0x42, 0x0a, 0x67, 0x6d,
		0x4b, 0x4a, 0x45, 0x51, 0x45, 0x6e, 0x47, 0x41, 0x42, 0x50,
		0x6b, 0x71, 0x73, 0x77, 0x73, 0x2f, 0x43, 0x54, 0x67, 0x6b,
		0x54, 0x61, 0x62, 0x37, 0x66, 0x4d, 0x64, 0x71, 0x45, 0x36,
		0x4f, 0x6b, 0x34, 0x77, 0x6e, 0x72, 0x32, 0x77, 0x50, 0x38,
		0x6d, 0x78, 0x36, 0x4c, 0x51, 0x6f, 0x2b, 0x6d, 0x7a, 0x5a,
		0x62, 0x62, 0x59, 0x2f, 0x48, 0x43, 0x32, 0x42, 0x63, 0x4c,
		0x70, 0x6f, 0x0a, 0x75, 0x66, 0x36, 0x32, 0x4f, 0x63, 0x49,
		0x46, 0x78, 0x68, 0x33, 0x43, 0x48, 0x31, 0x4f, 0x66, 0x59,
		0x45, 0x4b, 0x7a, 0x7a, 0x6b, 0x61, 0x2f, 0x70, 0x6b, 0x49,
		0x68, 0x38, 0x35, 0x51, 0x51, 0x75, 0x66, 0x34, 0x38, 0x5a,
		0x5a, 0x64, 0x52, 0x4d, 0x6a, 0x32, 0x77, 0x34, 0x4a, 0x4d,
		0x59, 0x70, 0x4b, 0x32, 0x4e, 0x49, 0x6e, 0x76, 0x6e, 0x54,
		0x4e, 0x55, 0x42, 0x68, 0x78, 0x46, 0x6c, 0x0a, 0x72, 0x73,
		0x68, 0x52, 0x68, 0x52, 0x57, 0x71, 0x45, 0x37, 0x53, 0x2b,
		0x5a, 0x57, 0x77, 0x68, 0x50, 0x69, 0x56, 0x6e, 0x4b, 0x61,
		0x61, 0x62, 0x33, 0x53, 0x6d, 0x41, 0x50, 0x57, 0x74, 0x4e,
		0x49, 0x41, 0x68, 0x72, 0x35, 0x62, 0x55, 0x43, 0x67, 0x67,
		0x45, 0x42, 0x41, 0x4f, 0x72, 0x31, 0x70, 0x74, 0x54, 0x78,
		0x65, 0x52, 0x45, 0x50, 0x48, 0x72, 0x42, 0x4b, 0x59, 0x76,
		0x36, 0x49, 0x0a, 0x45, 0x55, 0x7a, 0x70, 0x76, 0x39, 0x4c,
		0x33, 0x71, 0x72, 0x5a, 0x65, 0x4b, 0x2f, 0x31, 0x43, 0x52,
		0x71, 0x49, 0x4e, 0x67, 0x4c, 0x39, 0x4d, 0x2f, 0x76, 0x57,
		0x78, 0x41, 0x6b, 0x62, 0x4a, 0x72, 0x65, 0x61, 0x30, 0x64,
		0x58, 0x39, 0x6b, 0x64, 0x74, 0x48, 0x77, 0x57, 0x6e, 0x42,
		0x37, 0x61, 0x53, 0x56, 0x61, 0x38, 0x38, 0x56, 0x6e, 0x4d,
		0x72, 0x6d, 0x49, 0x61, 0x4f, 0x33, 0x55, 0x0a, 0x52, 0x71,
		0x58, 0x74, 0x42, 0x65, 0x77, 0x75, 0x63, 0x54, 0x50, 0x69,
		0x4f, 0x54, 0x67, 0x46, 0x61, 0x35, 0x35, 0x64, 0x35, 0x74,
		0x38, 0x53, 0x32, 0x57, 0x34, 0x4d, 0x36, 0x59, 0x4f, 0x41,
		0x6c, 0x54, 0x74, 0x52, 0x67, 0x32, 0x2f, 0x4c, 0x6f, 0x76,
		0x49, 0x68, 0x63, 0x48, 0x38, 0x4d, 0x38, 0x76, 0x79, 0x6a,
		0x67, 0x55, 0x78, 0x61, 0x35, 0x34, 0x38, 0x6a, 0x6f, 0x47,
		0x44, 0x73, 0x0a, 0x6c, 0x74, 0x4d, 0x42, 0x6d, 0x65, 0x31,
		0x4c, 0x6f, 0x2f, 0x72, 0x50, 0x44, 0x64, 0x73, 0x55, 0x4a,
		0x50, 0x6c, 0x34, 0x4d, 0x36, 0x33, 0x59, 0x42, 0x75, 0x54,
		0x78, 0x78, 0x71, 0x78, 0x36, 0x34, 0x4c, 0x73, 0x55, 0x4b,
		0x4b, 0x44, 0x72, 0x76, 0x44, 0x7a, 0x66, 0x2f, 0x64, 0x48,
		0x56, 0x39, 0x61, 0x79, 0x41, 0x50, 0x75, 0x31, 0x44, 0x39,
		0x73, 0x43, 0x77, 0x38, 0x65, 0x39, 0x6e, 0x0a, 0x4e, 0x39,
		0x44, 0x67, 0x71, 0x31, 0x72, 0x71, 0x49, 0x53, 0x5a, 0x38,
		0x78, 0x6e, 0x76, 0x58, 0x48, 0x46, 0x4c, 0x72, 0x4e, 0x51,
		0x67, 0x49, 0x53, 0x75, 0x67, 0x38, 0x2f, 0x72, 0x39, 0x78,
		0x39, 0x72, 0x72, 0x45, 0x74, 0x47, 0x4c, 0x6b, 0x49, 0x35,
		0x63, 0x6a, 0x45, 0x67, 0x35, 0x74, 0x61, 0x51, 0x52, 0x67,
		0x68, 0x62, 0x76, 0x7a, 0x69, 0x4a, 0x74, 0x49, 0x75, 0x6f,
		0x64, 0x47, 0x0a, 0x38, 0x37, 0x32, 0x49, 0x42, 0x73, 0x35,
		0x57, 0x44, 0x53, 0x53, 0x6a, 0x2f, 0x4c, 0x71, 0x2f, 0x59,
		0x5a, 0x4e, 0x73, 0x66, 0x79, 0x45, 0x66, 0x5a, 0x48, 0x6d,
		0x64, 0x31, 0x35, 0x54, 0x6b, 0x38, 0x4f, 0x74, 0x65, 0x69,
		0x36, 0x73, 0x66, 0x45, 0x71, 0x4a, 0x4e, 0x59, 0x36, 0x68,
		0x2b, 0x33, 0x56, 0x4c, 0x55, 0x41, 0x53, 0x68, 0x2b, 0x4b,
		0x62, 0x6c, 0x4b, 0x68, 0x79, 0x59, 0x7a, 0x0a, 0x6e, 0x32,
		0x30, 0x43, 0x67, 0x67, 0x45, 0x42, 0x41, 0x4b, 0x4b, 0x74,
		0x6b, 0x72, 0x6d, 0x4f, 0x49, 0x6b, 0x74, 0x31, 0x49, 0x76,
		0x57, 0x32, 0x6a, 0x47, 0x73, 0x68, 0x38, 0x35, 0x62, 0x53,
		0x33, 0x79, 0x74, 0x69, 0x66, 0x4e, 0x33, 0x36, 0x50, 0x2b,
		0x34, 0x39, 0x43, 0x6e, 0x47, 0x6d, 0x4c, 0x4a, 0x79, 0x4a,
		0x57, 0x31, 0x5a, 0x62, 0x79, 0x6c, 0x68, 0x31, 0x42, 0x75,
		0x70, 0x6c, 0x0a, 0x2b, 0x6d, 0x67, 0x77, 0x4c, 0x70, 0x6a,
		0x49, 0x55, 0x4c, 0x4b, 0x65, 0x2b, 0x55, 0x54, 0x78, 0x33,
		0x55, 0x46, 0x51, 0x42, 0x45, 0x53, 0x73, 0x46, 0x35, 0x38,
		0x6c, 0x54, 0x75, 0x35, 0x4e, 0x71, 0x4e, 0x6c, 0x33, 0x36,
		0x49, 0x55, 0x6b, 0x79, 0x47, 0x37, 0x4a, 0x4f, 0x55, 0x76,
		0x54, 0x41, 0x6d, 0x61, 0x2f, 0x6b, 0x49, 0x41, 0x65, 0x6d,
		0x2b, 0x58, 0x79, 0x79, 0x70, 0x6b, 0x59, 0x0a, 0x30, 0x34,
		0x7a, 0x4b, 0x57, 0x77, 0x35, 0x34, 0x64, 0x64, 0x33, 0x37,
		0x4b, 0x70, 0x4a, 0x74, 0x6b, 0x49, 0x67, 0x6d, 0x56, 0x45,
		0x71, 0x51, 0x49, 0x37, 0x78, 0x7a, 0x74, 0x45, 0x74, 0x4d,
		0x55, 0x55, 0x54, 0x6f, 0x62, 0x58, 0x31, 0x54, 0x5a, 0x6a,
		0x67, 0x31, 0x2f, 0x4c, 0x43, 0x41, 0x4f, 0x2b, 0x53, 0x5a,
		0x44, 0x59, 0x4a, 0x73, 0x55, 0x2f, 0x7a, 0x39, 0x33, 0x47,
		0x36, 0x7a, 0x0a, 0x38, 0x61, 0x31, 0x46, 0x6e, 0x48, 0x54,
		0x46, 0x6e, 0x37, 0x44, 0x49, 0x5a, 0x37, 0x58, 0x75, 0x47,
		0x2b, 0x37, 0x65, 0x48, 0x41, 0x50, 0x2b, 0x38, 0x5a, 0x45,
		0x71, 0x62, 0x72, 0x48, 0x49, 0x37, 0x2f, 0x61, 0x6f, 0x50,
		0x67, 0x58, 0x41, 0x69, 0x6f, 0x67, 0x72, 0x75, 0x2b, 0x66,
		0x50, 0x73, 0x44, 0x4b, 0x4e, 0x4e, 0x69, 0x68, 0x38, 0x6e,
		0x30, 0x34, 0x31, 0x47, 0x6d, 0x6b, 0x58, 0x0a, 0x54, 0x45,
		0x4c, 0x52, 0x2b, 0x54, 0x31, 0x7a, 0x51, 0x6c, 0x36, 0x63,
		0x59, 0x74, 0x68, 0x6f, 0x58, 0x72, 0x4a, 0x7a, 0x6a, 0x68,
		0x78, 0x36, 0x73, 0x6e, 0x34, 0x78, 0x33, 0x75, 0x4a, 0x74,
		0x2f, 0x63, 0x66, 0x73, 0x34, 0x31, 0x63, 0x72, 0x5a, 0x4f,
		0x59, 0x35, 0x41, 0x4c, 0x69, 0x31, 0x6d, 0x32, 0x30, 0x76,
		0x5a, 0x71, 0x59, 0x6c, 0x49, 0x45, 0x57, 0x50, 0x41, 0x77,
		0x34, 0x4b, 0x0a, 0x77, 0x64, 0x45, 0x44, 0x47, 0x64, 0x37,
		0x70, 0x6d, 0x61, 0x4c, 0x77, 0x42, 0x72, 0x58, 0x2b, 0x49,
		0x55, 0x69, 0x44, 0x75, 0x7a, 0x46, 0x30, 0x34, 0x41, 0x34,
		0x6a, 0x31, 0x50, 0x67, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d,
		0x2d, 0x45, 0x4e, 0x44, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41,
		0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d,
		0x2d, 0x0a,
	}},
};
/*

ECDSAP256SHA256 (256b)

ecdsa256.	DNSKEY	  256  3  13  8uD7C4THTM/w7uhryRSToeE/jKT78/p853RX0L5EwrZrSLBubLPiBw7gbvUP6SsIga5ZQ4CSAxNmYA/gZsuXzA==
ecdsa256.	DS    	 5345  13  1  528395F3CDCAA07AD315A9B1D05024E1378C2BAA
ecdsa256.	DS    	 5345  13  2  6D7D14EBC1BEE4FA8D394D0A1AC0663399114E608DCA4D53DD4A683DA390017B
ecdsa256.	DS    	 5345  13  4  9D887533DFC828CC5B04CA9B745991D556DCB9A0CAEA5901FEC1CA3C0DD7348CE8E6DD997CC80AA7A9BEDBB9CB653002

-----BEGIN PRIVATE KEY-----
MIGTAgEAMBMGByqGSM49AgEGCCqGSM49AwEHBHkwdwIBAQQgiyLIPdk3DOIxVmmS
YlmTstbtUPiVlEyDX46psyCwNVSgCgYIKoZIzj0DAQehRANCAATy4PsLhMdMz/Du
6GvJFJOh4T+MpPvz+nzndFfQvkTCtmtIsG5ss+IHDuBu9Q/pKwiBrllDgJIDE2Zg
D+Bmy5fM
-----END PRIVATE KEY-----

*/
static const key_parameters_t SAMPLE_ECDSA_P256_SHA256_KEY = {
	.name = (uint8_t *)"\x08""ecdsa256",
	.flags = 256,
	.protocol = 3,
	.algorithm = 13,
	.public_key = { .size = 64, .data = (uint8_t []){
		0xf2, 0xe0, 0xfb, 0x0b, 0x84, 0xc7, 0x4c, 0xcf, 0xf0, 0xee,
		0xe8, 0x6b, 0xc9, 0x14, 0x93, 0xa1, 0xe1, 0x3f, 0x8c, 0xa4,
		0xfb, 0xf3, 0xfa, 0x7c, 0xe7, 0x74, 0x57, 0xd0, 0xbe, 0x44,
		0xc2, 0xb6, 0x6b, 0x48, 0xb0, 0x6e, 0x6c, 0xb3, 0xe2, 0x07,
		0x0e, 0xe0, 0x6e, 0xf5, 0x0f, 0xe9, 0x2b, 0x08, 0x81, 0xae,
		0x59, 0x43, 0x80, 0x92, 0x03, 0x13, 0x66, 0x60, 0x0f, 0xe0,
		0x66, 0xcb, 0x97, 0xcc,
	}},
	.rdata = { .size = 68, .data = (uint8_t []){
		01, 0x00, 0x03, 0x0d,
		0xf2, 0xe0, 0xfb, 0x0b, 0x84, 0xc7, 0x4c, 0xcf, 0xf0, 0xee,
		0xe8, 0x6b, 0xc9, 0x14, 0x93, 0xa1, 0xe1, 0x3f, 0x8c, 0xa4,
		0xfb, 0xf3, 0xfa, 0x7c, 0xe7, 0x74, 0x57, 0xd0, 0xbe, 0x44,
		0xc2, 0xb6, 0x6b, 0x48, 0xb0, 0x6e, 0x6c, 0xb3, 0xe2, 0x07,
		0x0e, 0xe0, 0x6e, 0xf5, 0x0f, 0xe9, 0x2b, 0x08, 0x81, 0xae,
		0x59, 0x43, 0x80, 0x92, 0x03, 0x13, 0x66, 0x60, 0x0f, 0xe0,
		0x66, 0xcb, 0x97, 0xcc,
	}},
	.key_id = "47fd10011e76cc6741af586041eae5519465fc8d",
	.keytag = 5345,
	.ds_sha1 = { .size = 24, .data = (uint8_t []){
		0x14, 0xe1, 0x0d, 0x01, 0x52, 0x83, 0x95, 0xf3, 0xcd, 0xca,
		0xa0, 0x7a, 0xd3, 0x15, 0xa9, 0xb1, 0xd0, 0x50, 0x24, 0xe1,
		0x37, 0x8c, 0x2b, 0xaa,
	}},
	.ds_sha256 = { .size = 36, .data = (uint8_t []){
		0x14, 0xe1, 0x0d, 0x02, 0x6d, 0x7d, 0x14, 0xeb, 0xc1, 0xbe,
		0xe4, 0xfa, 0x8d, 0x39, 0x4d, 0x0a, 0x1a, 0xc0, 0x66, 0x33,
		0x99, 0x11, 0x4e, 0x60, 0x8d, 0xca, 0x4d, 0x53, 0xdd, 0x4a,
		0x68, 0x3d, 0xa3, 0x90, 0x01, 0x7b,
	}},
	.ds_sha384 = { .size = 52, .data = (uint8_t []){
		0x14, 0xe1, 0x0d, 0x04, 0x9d, 0x88, 0x75, 0x33, 0xdf, 0xc8,
		0x28, 0xcc, 0x5b, 0x04, 0xca, 0x9b, 0x74, 0x59, 0x91, 0xd5,
		0x56, 0xdc, 0xb9, 0xa0, 0xca, 0xea, 0x59, 0x01, 0xfe, 0xc1,
		0xca, 0x3c, 0x0d, 0xd7, 0x34, 0x8c, 0xe8, 0xe6, 0xdd, 0x99,
		0x7c, 0xc8, 0x0a, 0xa7, 0xa9, 0xbe, 0xdb, 0xb9, 0xcb, 0x65,
		0x30, 0x02,
	}},
	.bit_size = 256,
	.pem = { .size = 258, .data = (uint8_t []){
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49,
		0x47, 0x54, 0x41, 0x67, 0x45, 0x41, 0x4d, 0x42, 0x4d, 0x47,
		0x42, 0x79, 0x71, 0x47, 0x53, 0x4d, 0x34, 0x39, 0x41, 0x67,
		0x45, 0x47, 0x43, 0x43, 0x71, 0x47, 0x53, 0x4d, 0x34, 0x39,
		0x41, 0x77, 0x45, 0x48, 0x42, 0x48, 0x6b, 0x77, 0x64, 0x77,
		0x49, 0x42, 0x41, 0x51, 0x51, 0x67, 0x69, 0x79, 0x4c, 0x49,
		0x50, 0x64, 0x6b, 0x33, 0x44, 0x4f, 0x49, 0x78, 0x56, 0x6d,
		0x6d, 0x53, 0x0a, 0x59, 0x6c, 0x6d, 0x54, 0x73, 0x74, 0x62,
		0x74, 0x55, 0x50, 0x69, 0x56, 0x6c, 0x45, 0x79, 0x44, 0x58,
		0x34, 0x36, 0x70, 0x73, 0x79, 0x43, 0x77, 0x4e, 0x56, 0x53,
		0x67, 0x43, 0x67, 0x59, 0x49, 0x4b, 0x6f, 0x5a, 0x49, 0x7a,
		0x6a, 0x30, 0x44, 0x41, 0x51, 0x65, 0x68, 0x52, 0x41, 0x4e,
		0x43, 0x41, 0x41, 0x54, 0x79, 0x34, 0x50, 0x73, 0x4c, 0x68,
		0x4d, 0x64, 0x4d, 0x7a, 0x2f, 0x44, 0x75, 0x0a, 0x36, 0x47,
		0x76, 0x4a, 0x46, 0x4a, 0x4f, 0x68, 0x34, 0x54, 0x2b, 0x4d,
		0x70, 0x50, 0x76, 0x7a, 0x2b, 0x6e, 0x7a, 0x6e, 0x64, 0x46,
		0x66, 0x51, 0x76, 0x6b, 0x54, 0x43, 0x74, 0x6d, 0x74, 0x49,
		0x73, 0x47, 0x35, 0x73, 0x73, 0x2b, 0x49, 0x48, 0x44, 0x75,
		0x42, 0x75, 0x39, 0x51, 0x2f, 0x70, 0x4b, 0x77, 0x69, 0x42,
		0x72, 0x6c, 0x6c, 0x44, 0x67, 0x4a, 0x49, 0x44, 0x45, 0x32,
		0x5a, 0x67, 0x0a, 0x44, 0x2b, 0x42, 0x6d, 0x79, 0x35, 0x66,
		0x4d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a,
	}},
};
/*

ECDSAP384SHA384 (384b)

ecdsa384.	DNSKEY	  256  3  14  7CeBkmqMzybnRq9hXt8OoIOw77/LYzDfXGNs6JqHcGIAGRkc6PK4hvf7E5c6CITslUfsSvjboh+fkXQeFvsfipcmB8KUp+YOPN3kfL8Im+kkP+NSnzMiQ/oEiLJEus6r
ecdsa384.	DS    	54424  14  1  676AEF30FAC81141962216BDD98B4EB7FB97BC86
ecdsa384.	DS    	54424  14  2  2C3104425F555242D46111A87C3A9D6657F88238F13B529CA890667967F388BA
ecdsa384.	DS    	54424  14  4  C7B98E368420DA934FC877B271F111E820D61F2D5E399FC6997E796FB924E2849049F7480A1C937C50BE0C41F57CB006

-----BEGIN PRIVATE KEY-----
MIG/AgEAMBAGByqGSM49AgEGBSuBBAAiBIGnMIGkAgEBBDC3fRdw4A1GE00ZN39v
wHrOhkY0Tce5J7VODdoTJo25yAAog8tyT6N2IkibgHm4iYmgBwYFK4EEACKhZANi
AATsJ4GSaozPJudGr2Fe3w6gg7Dvv8tjMN9cY2zomodwYgAZGRzo8riG9/sTlzoI
hOyVR+xK+NuiH5+RdB4W+x+KlyYHwpSn5g483eR8vwib6SQ/41KfMyJD+gSIskS6
zqs=
-----END PRIVATE KEY-----

*/
static const key_parameters_t SAMPLE_ECDSA_P384_SHA384_KEY = {
	.name = (uint8_t *)"\x08""ecdsa384",
	.flags = 256,
	.protocol = 3,
	.algorithm = 14,
	.public_key = { .size = 96, .data = (uint8_t []){
		0xec, 0x27, 0x81, 0x92, 0x6a, 0x8c, 0xcf, 0x26, 0xe7, 0x46,
		0xaf, 0x61, 0x5e, 0xdf, 0x0e, 0xa0, 0x83, 0xb0, 0xef, 0xbf,
		0xcb, 0x63, 0x30, 0xdf, 0x5c, 0x63, 0x6c, 0xe8, 0x9a, 0x87,
		0x70, 0x62, 0x00, 0x19, 0x19, 0x1c, 0xe8, 0xf2, 0xb8, 0x86,
		0xf7, 0xfb, 0x13, 0x97, 0x3a, 0x08, 0x84, 0xec, 0x95, 0x47,
		0xec, 0x4a, 0xf8, 0xdb, 0xa2, 0x1f, 0x9f, 0x91, 0x74, 0x1e,
		0x16, 0xfb, 0x1f, 0x8a, 0x97, 0x26, 0x07, 0xc2, 0x94, 0xa7,
		0xe6, 0x0e, 0x3c, 0xdd, 0xe4, 0x7c, 0xbf, 0x08, 0x9b, 0xe9,
		0x24, 0x3f, 0xe3, 0x52, 0x9f, 0x33, 0x22, 0x43, 0xfa, 0x04,
		0x88, 0xb2, 0x44, 0xba, 0xce, 0xab,
	}},
	.rdata = { .size = 100, .data = (uint8_t []){
		01, 0x00, 0x03, 0x0e,
		0xec, 0x27, 0x81, 0x92, 0x6a, 0x8c, 0xcf, 0x26, 0xe7, 0x46,
		0xaf, 0x61, 0x5e, 0xdf, 0x0e, 0xa0, 0x83, 0xb0, 0xef, 0xbf,
		0xcb, 0x63, 0x30, 0xdf, 0x5c, 0x63, 0x6c, 0xe8, 0x9a, 0x87,
		0x70, 0x62, 0x00, 0x19, 0x19, 0x1c, 0xe8, 0xf2, 0xb8, 0x86,
		0xf7, 0xfb, 0x13, 0x97, 0x3a, 0x08, 0x84, 0xec, 0x95, 0x47,
		0xec, 0x4a, 0xf8, 0xdb, 0xa2, 0x1f, 0x9f, 0x91, 0x74, 0x1e,
		0x16, 0xfb, 0x1f, 0x8a, 0x97, 0x26, 0x07, 0xc2, 0x94, 0xa7,
		0xe6, 0x0e, 0x3c, 0xdd, 0xe4, 0x7c, 0xbf, 0x08, 0x9b, 0xe9,
		0x24, 0x3f, 0xe3, 0x52, 0x9f, 0x33, 0x22, 0x43, 0xfa, 0x04,
		0x88, 0xb2, 0x44, 0xba, 0xce, 0xab,
	}},
	.key_id = "93e55ee1bc62a4922a4872b7b4416afa8788964e",
	.keytag = 54424,
	.ds_sha1 = { .size = 24, .data = (uint8_t []){
		0xd4, 0x98, 0x0e, 0x01, 0x67, 0x6a, 0xef, 0x30, 0xfa, 0xc8,
		0x11, 0x41, 0x96, 0x22, 0x16, 0xbd, 0xd9, 0x8b, 0x4e, 0xb7,
		0xfb, 0x97, 0xbc, 0x86,
	}},
	.ds_sha256 = { .size = 36, .data = (uint8_t []){
		0xd4, 0x98, 0x0e, 0x02, 0x2c, 0x31, 0x04, 0x42, 0x5f, 0x55,
		0x52, 0x42, 0xd4, 0x61, 0x11, 0xa8, 0x7c, 0x3a, 0x9d, 0x66,
		0x57, 0xf8, 0x82, 0x38, 0xf1, 0x3b, 0x52, 0x9c, 0xa8, 0x90,
		0x66, 0x79, 0x67, 0xf3, 0x88, 0xba,
	}},
	.ds_sha384 = { .size = 52, .data = (uint8_t []){
		0xd4, 0x98, 0x0e, 0x04, 0xc7, 0xb9, 0x8e, 0x36, 0x84, 0x20,
		0xda, 0x93, 0x4f, 0xc8, 0x77, 0xb2, 0x71, 0xf1, 0x11, 0xe8,
		0x20, 0xd6, 0x1f, 0x2d, 0x5e, 0x39, 0x9f, 0xc6, 0x99, 0x7e,
		0x79, 0x6f, 0xb9, 0x24, 0xe2, 0x84, 0x90, 0x49, 0xf7, 0x48,
		0x0a, 0x1c, 0x93, 0x7c, 0x50, 0xbe, 0x0c, 0x41, 0xf5, 0x7c,
		0xb0, 0x06,
	}},
	.bit_size = 384,
	.pem = { .size = 319, .data = (uint8_t []){
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49,
		0x47, 0x2f, 0x41, 0x67, 0x45, 0x41, 0x4d, 0x42, 0x41, 0x47,
		0x42, 0x79, 0x71, 0x47, 0x53, 0x4d, 0x34, 0x39, 0x41, 0x67,
		0x45, 0x47, 0x42, 0x53, 0x75, 0x42, 0x42, 0x41, 0x41, 0x69,
		0x42, 0x49, 0x47, 0x6e, 0x4d, 0x49, 0x47, 0x6b, 0x41, 0x67,
		0x45, 0x42, 0x42, 0x44, 0x43, 0x33, 0x66, 0x52, 0x64, 0x77,
		0x34, 0x41, 0x31, 0x47, 0x45, 0x30, 0x30, 0x5a, 0x4e, 0x33,
		0x39, 0x76, 0x0a, 0x77, 0x48, 0x72, 0x4f, 0x68, 0x6b, 0x59,
		0x30, 0x54, 0x63, 0x65, 0x35, 0x4a, 0x37, 0x56, 0x4f, 0x44,
		0x64, 0x6f, 0x54, 0x4a, 0x6f, 0x32, 0x35, 0x79, 0x41, 0x41,
		0x6f, 0x67, 0x38, 0x74, 0x79, 0x54, 0x36, 0x4e, 0x32, 0x49,
		0x6b, 0x69, 0x62, 0x67, 0x48, 0x6d, 0x34, 0x69, 0x59, 0x6d,
		0x67, 0x42, 0x77, 0x59, 0x46, 0x4b, 0x34, 0x45, 0x45, 0x41,
		0x43, 0x4b, 0x68, 0x5a, 0x41, 0x4e, 0x69, 0x0a, 0x41, 0x41,
		0x54, 0x73, 0x4a, 0x34, 0x47, 0x53, 0x61, 0x6f, 0x7a, 0x50,
		0x4a, 0x75, 0x64, 0x47, 0x72, 0x32, 0x46, 0x65, 0x33, 0x77,
		0x36, 0x67, 0x67, 0x37, 0x44, 0x76, 0x76, 0x38, 0x74, 0x6a,
		0x4d, 0x4e, 0x39, 0x63, 0x59, 0x32, 0x7a, 0x6f, 0x6d, 0x6f,
		0x64, 0x77, 0x59, 0x67, 0x41, 0x5a, 0x47, 0x52, 0x7a, 0x6f,
		0x38, 0x72, 0x69, 0x47, 0x39, 0x2f, 0x73, 0x54, 0x6c, 0x7a,
		0x6f, 0x49, 0x0a, 0x68, 0x4f, 0x79, 0x56, 0x52, 0x2b, 0x78,
		0x4b, 0x2b, 0x4e, 0x75, 0x69, 0x48, 0x35, 0x2b, 0x52, 0x64,
		0x42, 0x34, 0x57, 0x2b, 0x78, 0x2b, 0x4b, 0x6c, 0x79, 0x59,
		0x48, 0x77, 0x70, 0x53, 0x6e, 0x35, 0x67, 0x34, 0x38, 0x33,
		0x65, 0x52, 0x38, 0x76, 0x77, 0x69, 0x62, 0x36, 0x53, 0x51,
		0x2f, 0x34, 0x31, 0x4b, 0x66, 0x4d, 0x79, 0x4a, 0x44, 0x2b,
		0x67, 0x53, 0x49, 0x73, 0x6b, 0x53, 0x36, 0x0a, 0x7a, 0x71,
		0x73, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e,
		0x44, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20,
		0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a,
	}},
};
/*

ED25519 (256b)

ed25519.	DNSKEY	  256  3  15  l02Woi0iS8Aa25FQkUd9RMzZHJpBoRQwAQEX1SxZJA4=
ed25519.	DS    	 3612  15  1  501249721E1F09A79D30D5C6C4DCA1DC1DA4ED5D
ed25519.	DS    	 3612  15  2  1B1C8766B2A96566FF196F77C0C4194AF86AAA109C5346FF60231A27D2B07AC0
ed25519.	DS    	 3612  15  4  D11831153AF4985EFBD0AE792C967EB4AFF3C35488DB95F7E2F85DCEC74AE8F59F9A72641798C91C67C675DB1D710C18

-----BEGIN PRIVATE KEY-----
MC4CAQAwBQYDK2VwBCIEIDgyMjYwMzg0NjI4MDgwMTIyNjQ1MTkwMjA0MTQyMjYy
-----END PRIVATE KEY-----

*/
static const key_parameters_t SAMPLE_ED25519_KEY = {
	.name = (uint8_t *)"\x07""ed25519",
	.flags = 256,
	.protocol = 3,
	.algorithm = 15,
	.public_key = { .size = 32, .data = (uint8_t []){
		0x97, 0x4d, 0x96, 0xa2, 0x2d, 0x22, 0x4b, 0xc0, 0x1a, 0xdb,
		0x91, 0x50, 0x91, 0x47, 0x7d, 0x44, 0xcc, 0xd9, 0x1c, 0x9a,
		0x41, 0xa1, 0x14, 0x30, 0x01, 0x01, 0x17, 0xd5, 0x2c, 0x59,
		0x24, 0x0e,
	}},
	.rdata = { .size = 36, .data = (uint8_t []){
		01, 0x00, 0x03, 0x0f,
		0x97, 0x4d, 0x96, 0xa2, 0x2d, 0x22, 0x4b, 0xc0, 0x1a, 0xdb,
		0x91, 0x50, 0x91, 0x47, 0x7d, 0x44, 0xcc, 0xd9, 0x1c, 0x9a,
		0x41, 0xa1, 0x14, 0x30, 0x01, 0x01, 0x17, 0xd5, 0x2c, 0x59,
		0x24, 0x0e,
	}},
	.key_id = "f34a0e533cde445fccaaa78401d496fc347d6cb6",
	.keytag = 3612,
	.ds_sha1 = { .size = 24, .data = (uint8_t []){
		0x0e, 0x1c, 0x0f, 0x01, 0x50, 0x12, 0x49, 0x72, 0x1e, 0x1f,
		0x09, 0xa7, 0x9d, 0x30, 0xd5, 0xc6, 0xc4, 0xdc, 0xa1, 0xdc,
		0x1d, 0xa4, 0xed, 0x5d,
	}},
	.ds_sha256 = { .size = 36, .data = (uint8_t []){
		0x0e, 0x1c, 0x0f, 0x02, 0x1b, 0x1c, 0x87, 0x66, 0xb2, 0xa9,
		0x65, 0x66, 0xff, 0x19, 0x6f, 0x77, 0xc0, 0xc4, 0x19, 0x4a,
		0xf8, 0x6a, 0xaa, 0x10, 0x9c, 0x53, 0x46, 0xff, 0x60, 0x23,
		0x1a, 0x27, 0xd2, 0xb0, 0x7a, 0xc0,
	}},
	.ds_sha384 = { .size = 52, .data = (uint8_t []){
		0x0e, 0x1c, 0x0f, 0x04, 0xd1, 0x18, 0x31, 0x15, 0x3a, 0xf4,
		0x98, 0x5e, 0xfb, 0xd0, 0xae, 0x79, 0x2c, 0x96, 0x7e, 0xb4,
		0xaf, 0xf3, 0xc3, 0x54, 0x88, 0xdb, 0x95, 0xf7, 0xe2, 0xf8,
		0x5d, 0xce, 0xc7, 0x4a, 0xe8, 0xf5, 0x9f, 0x9a, 0x72, 0x64,
		0x17, 0x98, 0xc9, 0x1c, 0x67, 0xc6, 0x75, 0xdb, 0x1d, 0x71,
		0x0c, 0x18,
	}},
	.bit_size = 256,
	.pem = { .size = 119, .data = (uint8_t []){
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x43,
		0x34, 0x43, 0x41, 0x51, 0x41, 0x77, 0x42, 0x51, 0x59, 0x44,
		0x4b, 0x32, 0x56, 0x77, 0x42, 0x43, 0x49, 0x45, 0x49, 0x44,
		0x67, 0x79, 0x4d, 0x6a, 0x59, 0x77, 0x4d, 0x7a, 0x67, 0x30,
		0x4e, 0x6a, 0x49, 0x34, 0x4d, 0x44, 0x67, 0x77, 0x4d, 0x54,
		0x49, 0x79, 0x4e, 0x6a, 0x51, 0x31, 0x4d, 0x54, 0x6b, 0x77,
		0x4d, 0x6a, 0x41, 0x30, 0x4d, 0x54, 0x51, 0x79, 0x4d, 0x6a,
		0x59, 0x79, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e,
		0x44, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20,
		0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a,
	}},
};
/*

ED448 (456b)

ed448.	DNSKEY	  256  3  16  3y0sCUZKI+DSyZQTTbgHOFppOzYz01iZQQVUWw9gCb1bLWNqqC/5qq2PL9Tc6YaK2vHthBtaY0iA
ed448.	DS    	28205  16  1  CE1D28452326E1A03F2FE7D81D167FC8EFD9C888
ed448.	DS    	28205  16  2  CD1AE915704D3B742E42924A747BC37C252B9E465E0DA1547F0D8826BA073B65
ed448.	DS    	28205  16  4  F1D7572154302A965707BE8E83C300F66521DEA5239135FCB61D6CD52E4FCD04CD0A8047276F8671C31CA2B9C8561BE8

-----BEGIN PRIVATE KEY-----
MEcCAQAwBQYDK2VxBDsEOVsk9cLaP+mD8n97xRET5bCqCUDZpPCIkCGrXLJGJgXC
Glz4mOxH4qva7fIhEXm/62vtdbd55WRXIA==
-----END PRIVATE KEY-----

*/
static const key_parameters_t SAMPLE_ED448_KEY = {
	.name = (uint8_t *)"\x05""ed448",
	.flags = 256,
	.protocol = 3,
	.algorithm = 16,
	.public_key = { .size = 57, .data = (uint8_t []){
		0xdf, 0x2d, 0x2c, 0x09, 0x46, 0x4a, 0x23, 0xe0, 0xd2, 0xc9,
		0x94, 0x13, 0x4d, 0xb8, 0x07, 0x38, 0x5a, 0x69, 0x3b, 0x36,
		0x33, 0xd3, 0x58, 0x99, 0x41, 0x05, 0x54, 0x5b, 0x0f, 0x60,
		0x09, 0xbd, 0x5b, 0x2d, 0x63, 0x6a, 0xa8, 0x2f, 0xf9, 0xaa,
		0xad, 0x8f, 0x2f, 0xd4, 0xdc, 0xe9, 0x86, 0x8a, 0xda, 0xf1,
		0xed, 0x84, 0x1b, 0x5a, 0x63, 0x48, 0x80,
	}},
	.rdata = { .size = 61, .data = (uint8_t []){
		01, 0x00, 0x03, 0x10,
		0xdf, 0x2d, 0x2c, 0x09, 0x46, 0x4a, 0x23, 0xe0, 0xd2, 0xc9,
		0x94, 0x13, 0x4d, 0xb8, 0x07, 0x38, 0x5a, 0x69, 0x3b, 0x36,
		0x33, 0xd3, 0x58, 0x99, 0x41, 0x05, 0x54, 0x5b, 0x0f, 0x60,
		0x09, 0xbd, 0x5b, 0x2d, 0x63, 0x6a, 0xa8, 0x2f, 0xf9, 0xaa,
		0xad, 0x8f, 0x2f, 0xd4, 0xdc, 0xe9, 0x86, 0x8a, 0xda, 0xf1,
		0xed, 0x84, 0x1b, 0x5a, 0x63, 0x48, 0x80,
	}},
	.key_id = "501a69b2d8ad46c721ffabaa9eaf8e7fa49c1454",
	.keytag = 28205,
	.ds_sha1 = { .size = 24, .data = (uint8_t []){
		0x6e, 0x2d, 0x10, 0x01, 0xce, 0x1d, 0x28, 0x45, 0x23, 0x26,
		0xe1, 0xa0, 0x3f, 0x2f, 0xe7, 0xd8, 0x1d, 0x16, 0x7f, 0xc8,
		0xef, 0xd9, 0xc8, 0x88,
	}},
	.ds_sha256 = { .size = 36, .data = (uint8_t []){
		0x6e, 0x2d, 0x10, 0x02, 0xcd, 0x1a, 0xe9, 0x15, 0x70, 0x4d,
		0x3b, 0x74, 0x2e, 0x42, 0x92, 0x4a, 0x74, 0x7b, 0xc3, 0x7c,
		0x25, 0x2b, 0x9e, 0x46, 0x5e, 0x0d, 0xa1, 0x54, 0x7f, 0x0d,
		0x88, 0x26, 0xba, 0x07, 0x3b, 0x65,
	}},
	.ds_sha384 = { .size = 52, .data = (uint8_t []){
		0x6e, 0x2d, 0x10, 0x04, 0xf1, 0xd7, 0x57, 0x21, 0x54, 0x30,
		0x2a, 0x96, 0x57, 0x07, 0xbe, 0x8e, 0x83, 0xc3, 0x00, 0xf6,
		0x65, 0x21, 0xde, 0xa5, 0x23, 0x91, 0x35, 0xfc, 0xb6, 0x1d,
		0x6c, 0xd5, 0x2e, 0x4f, 0xcd, 0x04, 0xcd, 0x0a, 0x80, 0x47,
		0x27, 0x6f, 0x86, 0x71, 0xc3, 0x1c, 0xa2, 0xb9, 0xc8, 0x56,
		0x1b, 0xe8,
	}},
	.bit_size = 456,
	.pem = { .size = 156, .data = (uint8_t []){
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e,
		0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
		0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x45,
		0x63, 0x43, 0x41, 0x51, 0x41, 0x77, 0x42, 0x51, 0x59, 0x44,
		0x4b, 0x32, 0x56, 0x78, 0x42, 0x44, 0x73, 0x45, 0x4f, 0x56,
		0x73, 0x6b, 0x39, 0x63, 0x4c, 0x61, 0x50, 0x2b, 0x6d, 0x44,
		0x38, 0x6e, 0x39, 0x37, 0x78, 0x52, 0x45, 0x54, 0x35, 0x62,
		0x43, 0x71, 0x43, 0x55, 0x44, 0x5a, 0x70, 0x50, 0x43, 0x49,
		0x6b, 0x43, 0x47, 0x72, 0x58, 0x4c, 0x4a, 0x47, 0x4a, 0x67,
		0x58, 0x43, 0x0a, 0x47, 0x6c, 0x7a, 0x34, 0x6d, 0x4f, 0x78,
		0x48, 0x34, 0x71, 0x76, 0x61, 0x37, 0x66, 0x49, 0x68, 0x45,
		0x58, 0x6d, 0x2f, 0x36, 0x32, 0x76, 0x74, 0x64, 0x62, 0x64,
		0x35, 0x35, 0x57, 0x52, 0x58, 0x49, 0x41, 0x3d, 0x3d, 0x0a,
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x50,
		0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59,
		0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a,
	}},
};
