/**
 * UGENE - Integrated Bioinformatics Tools.
 * Copyright (C) 2008-2025 UniPro <ugene@unipro.ru>
 * http://ugene.net
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include "ProjectTreeItemSelectorDialogFiller.h"
#include <drivers/GTKeyboardDriver.h>
#include <primitives/GTTreeView.h>
#include <primitives/GTWidget.h>

#include <QApplication>
#include <QDialogButtonBox>
#include <QTreeWidget>

#include <U2Core/U2IdTypes.h>

#include "GTUtilsProjectTreeView.h"

namespace U2 {
using namespace HI;

#define GT_CLASS_NAME "ProjectTreeItemSelectorDialogFiller"

ProjectTreeItemSelectorDialogFiller::ProjectTreeItemSelectorDialogFiller(
    const QString& documentName,
    const QString& objectName,
    const QSet<GObjectType>& acceptableTypes,
    SelectionMode mode,
    int expectedDocCount)
    : Filler("ProjectTreeItemSelectorDialogBase"), acceptableTypes(acceptableTypes), mode(mode), expectedDocCount(expectedDocCount) {
    itemsToSelect.insert(documentName, {objectName});
}

ProjectTreeItemSelectorDialogFiller::ProjectTreeItemSelectorDialogFiller(
    const QMap<QString, QStringList>& itemsToSelect,
    const QSet<GObjectType>& acceptableTypes,
    SelectionMode mode,
    int expectedDocCount)
    : Filler("ProjectTreeItemSelectorDialogBase"), itemsToSelect(itemsToSelect), acceptableTypes(acceptableTypes), mode(mode),
      expectedDocCount(expectedDocCount) {
}

ProjectTreeItemSelectorDialogFiller::ProjectTreeItemSelectorDialogFiller(CustomScenario* scenario)
    : Filler("ProjectTreeItemSelectorDialogBase", scenario),
      mode(Single),
      expectedDocCount(0) {
}

static bool checkTreeRowCount(QTreeView* tree, int expectedDocCount) {
    int visibleItemCount = 0;
    for (int i = 0; i < tree->model()->rowCount(); ++i) {
        Qt::ItemFlags itemFlags = tree->model()->flags(tree->model()->index(i, 0));
        if (itemFlags != Qt::NoItemFlags) {
            visibleItemCount++;
        }
    }
    return visibleItemCount == expectedDocCount;
}

void ProjectTreeItemSelectorDialogFiller::commonScenario() {
    auto dialog = GTWidget::getActiveModalWidget();
    auto treeView = GTWidget::findTreeView("treeView", dialog);
    CHECK_SET_ERR(expectedDocCount == -1 || checkTreeRowCount(treeView, expectedDocCount), "Unexpected document count");

    GTGlobals::FindOptions options;
    options.depth = GTGlobals::FindOptions::INFINITE_DEPTH;

    bool isFirstClick = true;
    QList<QString> allItemKeys = itemsToSelect.keys();
    auto getCurrentClickModifier = [this, &isFirstClick] { return isFirstClick ? Qt::Key_unknown : (mode == Continuous ? Qt::Key_Shift : Qt::Key_Control); };
    for (const QString& itemKey : qAsConst(allItemKeys)) {
        QModelIndex parentItemIndex = GTUtilsProjectTreeView::findIndex(treeView, itemKey, options);
        if (!acceptableTypes.isEmpty()) {
            GTUtilsProjectTreeView::checkObjectTypes(treeView, acceptableTypes, parentItemIndex);
        }
        QStringList objectNames = itemsToSelect.value(itemKey);
        if (objectNames.isEmpty()) {  // Select the document itself.
            GTTreeView::click(treeView, parentItemIndex, getCurrentClickModifier());
            isFirstClick = false;
            continue;
        }
        for (const QString& objectName : qAsConst(objectNames)) {
            QModelIndex objectIndex = GTUtilsProjectTreeView::findIndex(treeView, objectName, parentItemIndex, options);
            GTTreeView::click(treeView, objectIndex, getCurrentClickModifier());
            isFirstClick = false;
        }
    }
    // Close the dialog.
    GTUtilsDialog::clickButtonBox(dialog, QDialogButtonBox::Ok);
}

#undef GT_CLASS_NAME

}  // namespace U2
