Currently libcpucycles supports the following cycle counters. Some
cycle counters are actually other forms of counters that libcpucycles
scales to imitate a cycle counter. There is
[separate documentation](selection.html)
for how libcpucycles makes a choice of cycle counter.
There is a `cpucycles-open` command that, when run as `root`, often
succeeds in enabling higher-quality cycle counters until reboot. See the
[security considerations](security.html) regarding enabling or disabling
counters and regarding overclocking (Turbo Boost, Turbo Core, etc.).

`amd64-perfpmc`: Requires a 64-bit Intel/AMD platform; the Linux
`perf_event` interface; `/proc/sys/kernel/perf_event_paranoid` at most 2;
any existing files among `/sys/devices/{cpu,cpu_core,cpu_atom}/rdpmc` at
least 2. Accesses a cycle counter through RDPMC. This counter runs at
the clock frequency of the CPU core.

`amd64-pmc`: Requires a 64-bit Intel/AMD platform and an OS that enables
user-level RDPMC access. Accesses a cycle counter through RDPMC. This
counter runs at the clock frequency of the CPU core.

`amd64-tsc`, `amd64-tscasm`: Requires a 64-bit Intel/AMD platform.
Requires RDTSC to be enabled, which it is by default. Uses RDTSC to
access the CPU's time-stamp counter. On current CPUs, this is an
off-core clock rather than a cycle counter, but it is typically a very
fast off-core clock, making it adequate for seeing cycle counts if
overclocking and underclocking are disabled. The difference between
`tsc` and `tscasm` is that `tsc` uses the compiler's `__rdtsc()` while
`tscasm` uses inline assembly.

`arm32-cortex`: Requires a 32-bit ARMv7-A platform. Uses
`mrc p15, 0, %0, c9, c13, 0` to read the cycle counter. Requires user
access to the cycle counter, which is not enabled by default but can (sometimes) be
enabled under Linux via
[a kernel module](https://github.com/thoughtpolice/enable_arm_pmu).
This counter is natively 32 bits, but libcpucycles watches how the
counter and `gettimeofday` increase to compute a 64-bit extension of the
counter.

`arm32-1176`: Requires a 32-bit ARM1176 platform. Uses
`mrc p15, 0, %0, c15, c12, 1` to read the cycle counter. Requires user
access to the cycle counter, which is not enabled by default but can (sometimes) be
enabled under Linux via
[a kernel module](https://bench.cr.yp.to/cpucycles/n810.html).
This counter is natively 32 bits, but libcpucycles watches how the
counter and `gettimeofday` increase to compute a 64-bit extension of the
counter.

`arm64-perfpmc`: Requires a 64-bit ARMv8-A platform; the Linux
`perf_event` interface; on current Linux kernels, 1 in
`/proc/sys/kernel/perf_user_access`; on older Linux kernels,
[a kernel module](https://github.com/rdolbeau/enable_arm_pmu)
to enable PMCCNTR access. Uses `mrs %0, PMCCNTR_EL0` to read the cycle
counter.

`arm64-pmc`: Requires a 64-bit ARMv8-A platform and an OS that enables
user-level PMCCNTR access, for example an older Linux kernel with
[a kernel module](https://github.com/rdolbeau/enable_arm_pmu).
Uses `mrs %0, PMCCNTR_EL0` to read the cycle counter.

`arm64-vct`: Requires a 64-bit ARMv8-A platform. Uses
`mrs %0, CNTVCT_EL0` to read a "virtual count" timer. This is an
off-core clock, typically running at 24MHz. Results are scaled by
libcpucycles.

`loong64-rdtime`: Requires a 64-bit LoongArch platform.
Uses `rdtime.d` to read the "constant frequency timer".
This is an off-core clock, typically running at 100MHz.

`mips64-cc`: Requires a 64-bit MIPS platform. (Maybe the same code would
also work as `mips32-cc`, but this has not been tested yet.) Uses RDHWR
to read the hardware cycle counter (hardware register 2 times a constant
scale factor in hardware register 3). This counter is natively 32 bits,
but libcpucycles watches how the counter and `gettimeofday` increase to
compute a 64-bit extension of the counter.

`ppc32-mftb`: Requires a 32-bit PowerPC platform. Uses `mftb` and
`mftbu` to read the "time base". This is an off-core clock, typically
running at 24MHz.

`ppc64-mftb`: Requires a 64-bit PowerPC platform. Uses `mftb` and
`mftbu` to read the "time base". This is an off-core clock, typically
running at 24MHz.

`riscv32-perfrdcycle`: Requires a 32-bit RISC-V platform; the Linux
`perf_event` interface; on current Linux kernels, 1 in
`/proc/sys/kernel/perf_user_access`. Uses `rdcycle` and `rdcycleh` to
read a cycle counter.

`riscv32-rdcycle`: Requires a 32-bit RISC-V platform and an OS that
enables user-level `rdcycle` access. Uses `rdcycle` and `rdcycleh` to
read a cycle counter.

`riscv64-perfrdcycle`: Requires a 64-bit RISC-V platform; the Linux
`perf_event` interface; on current Linux kernels, 1 in
`/proc/sys/kernel/perf_user_access`. Uses `rdcycle` to read a cycle
counter.

`riscv64-rdcycle`: Requires a 64-bit RISC-V platform and an OS that
enables user-level `rdcycle` access. Uses `rdcycle` to read a cycle
counter.

`s390x-stckf`: Requires a 64-bit z/Architecture platform. Uses `stckf`
to read the TOD clock, which is documented to run at 4096MHz. On the
z15, this looks like a doubling of an off-core 2048MHz clock. Results
are scaled by libcpucycles.

`sparc64-rdtick`: Requires a 64-bit SPARC platform. Uses `rd %tick`
to read a cycle counter.

`x86-tsc`, `x86-tscasm`: Same as `amd64-tsc` and `amd64-tscasm`, but
for 32-bit Intel/AMD platforms instead of 64-bit Intel/AMD platforms.

`default-gettimeofday`: Reasonably portable. Resolution is limited to 1
microsecond. Results are scaled by libcpucycles.

`default-mach`: Requires an OS with `mach_absolute_time()`. Typically
runs at 24MHz. Results are scaled by libcpucycles.

`default-monotonic`: Requires `CLOCK_MONOTONIC`. Reasonably portable,
although might fail on older systems where `default-gettimeofday` works.
Resolution is limited to 1 nanosecond. Can be almost as good as a cycle
counter, or orders of magnitude worse, depending on the OS and CPU.
Results are scaled by libcpucycles.

`default-perfevent`: Requires the Linux `perf_event` interface, and a
CPU where `perf_event` supports `PERF_COUNT_HW_CPU_CYCLES`. Similar
variations in quality to `default-monotonic`, without the 1-nanosecond
limitation.

`default-zero`: The horrifying last resort if nothing else works.

## Examples

These are examples of `cpucycles-info` output on various machines. The
machines named `cfarm*` are from the
[GCC Compile Farm](https://gcc.gnu.org/wiki/CompileFarm).

A `median` line saying, e.g., `47 +47+28+0+2-5+0+2-5...` means that the
differences between adjacent cycle counts were 47+47, 47+28, 47+0, 47+2,
47−5, 47+0, 47+2, 47−5, etc., with median difference 47. The first few
differences are typically larger because of cache effects.
Current versions of libcpucycles use `iqm` (interquartile mean)
instead of `median`.

`berry0`,
Broadcom BCM2835:
```
cpucycles version 20240114
cpucycles tracesetup 0 arm32-cortex precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 arm32-1176 precision 22 scaling 1.000000 only32 1
cpucycles tracesetup 2 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 1199 scaling 1.000000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 1200 scaling 1000.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1000000000
cpucycles implementation arm32-1176
cpucycles median 720 +942+124+1+0+2+0+0+0+0+0+0+0+0+0+0+0+0+0+0+1+2+0+0+2+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+222+300+1+0+0+2+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 798307692...2045181819 with 1024 loops 12 microseconds
cpucycles observed persecond 915478260...1260523810 with 2048 loops 22 microseconds
cpucycles observed persecond 947809523...1106100000 with 4096 loops 41 microseconds
cpucycles observed persecond 966353658...1129037500 with 8192 loops 81 microseconds
cpucycles observed persecond 988490566...1030019109 with 16384 loops 158 microseconds
cpucycles observed persecond 995169327...1002034063 with 32768 loops 2379 microseconds
cpucycles observed persecond 996871019...1012568691 with 65536 loops 627 microseconds
cpucycles observed persecond 997832134...1004212170 with 131072 loops 1250 microseconds
cpucycles observed persecond 997740918...1000887780 with 262144 loops 5009 microseconds
cpucycles observed persecond 998528349...1001961164 with 524288 loops 5537 microseconds
cpucycles observed persecond 999202882...1001166794 with 1048576 loops 10547 microseconds
```

`berry2`,
Broadcom BCM2836:
```
cpucycles version 20251226
cpucycles tracesetup 0 arm32-cortex precision 310 scaling 1.000000 only32 1
cpucycles tracesetup 1 arm32-1176 precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 254 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 814 scaling 0.900000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 1200 scaling 900.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 900000000
cpucycles implementation default-perfevent
cpucycles iqm 154 +208+9+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+173+172+29+0-4+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 218360000...420565218 with 1024 loops 24 microseconds
cpucycles observed persecond 327781250...402566667 with 2048 loops 31 microseconds
cpucycles observed persecond 423204081...468085107 with 4096 loops 48 microseconds
cpucycles observed persecond 490369047...517231708 with 8192 loops 83 microseconds
cpucycles observed persecond 544033112...560208054 with 16384 loops 150 microseconds
cpucycles observed persecond 450227397...457187328 with 32768 loops 364 microseconds
cpucycles observed persecond 584518716...588967800 with 65536 loops 560 microseconds
cpucycles observed persecond 592214995...594508598 with 131072 loops 1106 microseconds
cpucycles observed persecond 595894545...597049591 with 262144 loops 2199 microseconds
cpucycles observed persecond 598008895...598558878 with 524288 loops 4383 microseconds
cpucycles observed persecond 595624105...595918873 with 1048576 loops 8802 microseconds
```

`pi3aplus`,
Broadcom BCM2837B0:
```
cpucycles version 20251226
cpucycles tracesetup 0 arm64-perfpmc precision 8 scaling 1.000000 only32 0
cpucycles tracesetup 1 arm64-pmc precision 0 scaling 1.000000 only32 0
cpucycles tracesetup 2 arm64-vct precision 173 scaling 73.000000 only32 0
cpucycles tracesetup 3 default-perfevent precision 191 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-monotonic precision 372 scaling 1.400000 only32 0
cpucycles tracesetup 6 default-gettimeofday precision 1700 scaling 1400.000000 only32 0
cpucycles tracesetup 7 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1400000000
cpucycles implementation arm64-perfpmc
cpucycles iqm 10 +209+13+3+8+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 1030250000...2148500000 with 2048 loops 3 microseconds
cpucycles observed persecond 1176285714...1705800000 with 4096 loops 6 microseconds
cpucycles observed persecond 1264076923...1519818182 with 8192 loops 12 microseconds
cpucycles observed persecond 1216777777...1324600000 with 16384 loops 26 microseconds
cpucycles observed persecond 1366187500...1431586957 with 32768 loops 47 microseconds
cpucycles observed persecond 1324939393...1355443299 with 65536 loops 98 microseconds
cpucycles observed persecond 1219818604...1232802817 with 131072 loops 214 microseconds
cpucycles observed persecond 1379857894...1387955027 with 262144 loops 379 microseconds
cpucycles observed persecond 1344465384...1348359898 with 524288 loops 779 microseconds
cpucycles observed persecond 1305948318...1307784913 with 1048576 loops 1605 microseconds
```

`pi5`,
Broadcom BCM2712,
Debian 12,
Linux kernel 6.12.25+rpt-rpi-2712,
running in power-saving mode
(`echo powersave > /sys/devices/system/cpu/cpufreq/policy0/scaling_governor`):
```
cpucycles version 20251226
cpucycles tracesetup 0 arm64-perfpmc precision 4 scaling 1.000000 only32 0
cpucycles tracesetup 1 arm64-pmc precision 0 scaling 1.000000 only32 0
cpucycles tracesetup 2 arm64-vct precision 144 scaling 44.500000 only32 0
cpucycles tracesetup 3 default-perfevent precision 158 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-monotonic precision 432 scaling 2.400000 only32 0
cpucycles tracesetup 6 default-gettimeofday precision 2700 scaling 2400.000000 only32 0
cpucycles tracesetup 7 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2400000000
cpucycles implementation arm64-perfpmc
cpucycles iqm 6 +0+1+4-4+1+2+1+2+0+3+0+3+0+3+0+3+0+3+1+4-1-1+5-4+7-3+8-4-1+4-3+6-3+10-4-4+6+0-4+7-3+8-4-4+7-3+8-4-4+7-3+8-4-2+5-3+12-4-4+7-3+8-4-4
cpucycles observed persecond 1176571428...1676600000 with 8192 loops 6 microseconds
cpucycles observed persecond 1369750000...1658400000 with 16384 loops 11 microseconds
cpucycles observed persecond 1426826086...1569238096 with 32768 loops 22 microseconds
cpucycles observed persecond 1457444444...1528418605 with 65536 loops 44 microseconds
cpucycles observed persecond 1473314606...1508655173 with 131072 loops 88 microseconds
cpucycles observed persecond 1489755681...1507614943 with 262144 loops 175 microseconds
cpucycles observed persecond 1498117142...1507094828 with 524288 loops 349 microseconds
cpucycles observed persecond 1498035714...1502511462 with 1048576 loops 699 microseconds
```

`bblack`,
TI Sitara XAM3359AZCZ100:
```
cpucycles version 20251226
cpucycles tracesetup 0 arm32-cortex precision 308 scaling 1.000000 only32 1
cpucycles tracesetup 1 arm32-1176 precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 1424 scaling 1.000000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 1300 scaling 1000.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1000000000
cpucycles implementation arm32-cortex
cpucycles iqm 1251 +1355+63+24+3-1+9+9-1+9-1-1+9-1-1-1+1+9+7-1-1-1-1-1-1-1-1-1+9-1-1-1+9-1-1-1+9-1+1+7-1-1-1-1-1-1-1-1-1-1+9-1-1-1-1-1+9-1+9+9+9-1+9+1+9
cpucycles observed persecond 702100000...3153750000 with 1024 loops 9 microseconds
cpucycles observed persecond 785666666...1626615385 with 2048 loops 14 microseconds
cpucycles observed persecond 837222222...4387840000 with 4096 loops 26 microseconds
cpucycles observed persecond 923108695...1106590910 with 8192 loops 45 microseconds
cpucycles observed persecond 957068965...1055564706 with 16384 loops 86 microseconds
cpucycles observed persecond 981711864...1019555556 with 32768 loops 235 microseconds
cpucycles observed persecond 987004210...1010238901 with 65536 loops 474 microseconds
cpucycles observed persecond 993804582...1010810811 with 131072 loops 741 microseconds
cpucycles observed persecond 996266338...1006985926 with 262144 loops 1422 microseconds
cpucycles observed persecond 997860246...1002197464 with 524288 loops 2761 microseconds
cpucycles observed persecond 999151671...1001252031 with 1048576 loops 9480 microseconds
```

`titan0`,
Intel Xeon E3-1275 V3:
```
cpucycles version 20251226
cpucycles tracesetup 0 amd64-perfpmc precision 37 scaling 1.000000 only32 0
cpucycles tracesetup 1 amd64-pmc precision 38 scaling 1.000000 only32 0
cpucycles tracesetup 2 amd64-tsc precision 124 scaling 1.000000 only32 0
cpucycles tracesetup 3 amd64-tscasm precision 124 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-perfevent precision 187 scaling 1.000000 only32 0
cpucycles tracesetup 5 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 6 default-monotonic precision 370 scaling 3.500000 only32 0
cpucycles tracesetup 7 default-gettimeofday precision 3800 scaling 3500.000000 only32 0
cpucycles tracesetup 8 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3500000000
cpucycles implementation amd64-perfpmc
cpucycles iqm 40 +22+21+21+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 2742166666...4160750000 with 16384 loops 5 microseconds
cpucycles observed persecond 2990727272...3678777778 with 32768 loops 10 microseconds
cpucycles observed persecond 3283350000...3658500000 with 65536 loops 19 microseconds
cpucycles observed persecond 3456500000...3653638889 with 131072 loops 37 microseconds
cpucycles observed persecond 3451986842...3547905406 with 262144 loops 75 microseconds
cpucycles observed persecond 3481086580...3512598254 with 524288 loops 230 microseconds
cpucycles observed persecond 3480681704...3498871537 with 1048576 loops 398 microseconds
```

`nucnuc`,
Intel Pentium N3700:
```
cpucycles version 20251226
cpucycles tracesetup 0 amd64-perfpmc precision 30 scaling 1.000000 only32 0
cpucycles tracesetup 1 amd64-pmc precision 30 scaling 1.000000 only32 0
cpucycles tracesetup 2 amd64-tsc precision 120 scaling 1.000000 only32 0
cpucycles tracesetup 3 amd64-tscasm precision 120 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-perfevent precision 438 scaling 1.000000 only32 0
cpucycles tracesetup 5 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 6 default-monotonic precision 420 scaling 1.600000 only32 0
cpucycles tracesetup 7 default-gettimeofday precision 1900 scaling 1600.000000 only32 0
cpucycles tracesetup 8 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1600000000
cpucycles implementation amd64-perfpmc
cpucycles iqm 58 +10+0+15+15+15+15+15+15+0+15+15+0+15+0+0+0+0+0+0+1+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 1059750000...2318000000 with 2048 loops 3 microseconds
cpucycles observed persecond 1391500000...2291500000 with 4096 loops 5 microseconds
cpucycles observed persecond 1376666666...1710900000 with 8192 loops 11 microseconds
cpucycles observed persecond 1495772727...1675250000 with 16384 loops 21 microseconds
cpucycles observed persecond 1563880952...1656900000 with 32768 loops 41 microseconds
cpucycles observed persecond 1580891566...1626938272 with 65536 loops 82 microseconds
cpucycles observed persecond 1589612121...1612582823 with 131072 loops 164 microseconds
cpucycles observed persecond 1593993920...1605532111 with 262144 loops 328 microseconds
cpucycles observed persecond 1592446745...1602621662 with 524288 loops 675 microseconds
cpucycles observed persecond 1598547256...1601419084 with 1048576 loops 1311 microseconds
```

`saber214`,
AMD FX-8350:
```
cpucycles version 20251226
cpucycles tracesetup 0 amd64-perfpmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 amd64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 amd64-tsc precision 171 scaling 1.000000 only32 0
cpucycles tracesetup 3 amd64-tscasm precision 170 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 6 default-monotonic precision 468 scaling 4.013176 only32 0
cpucycles tracesetup 7 default-gettimeofday precision 4313 scaling 4013.176000 only32 0
cpucycles tracesetup 8 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 4013176000
cpucycles implementation amd64-tscasm
cpucycles iqm 73 +43-3+3249-1+6+7+2+64+10+533-1+7-2-2+5+2-3+11+3-3-2+12+534+4545+21+6+125+269+4589+24+3-3-1-1+2+4+7-1-2-2-1-2-2-2-2-1-2-2-2-2-1-2-2-2-1-2-2-2-2-1-2-2-2-1
cpucycles observed persecond 3017500000...6235500000 with 4096 loops 3 microseconds
cpucycles observed persecond 3440714285...4897800000 with 8192 loops 6 microseconds
cpucycles observed persecond 3699384615...4411090910 with 16384 loops 12 microseconds
cpucycles observed persecond 3842000000...4195434783 with 32768 loops 24 microseconds
cpucycles observed persecond 3917469387...4093872341 with 65536 loops 48 microseconds
cpucycles observed persecond 3997593750...4087478724 with 131072 loops 95 microseconds
cpucycles observed persecond 3996843750...4041331579 with 262144 loops 191 microseconds
cpucycles observed persecond 4006895561...4029115486 with 524288 loops 382 microseconds
cpucycles observed persecond 4008774522...4019639847 with 1048576 loops 784 microseconds
```

`cfarm14`,
Intel Xeon E5-2620 v3,
Debian 12,
Linux kernel 6.1.0-26-amd64:
```
cpucycles version 20251226
cpucycles tracesetup 0 amd64-perfpmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 amd64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 amd64-tsc precision 118 scaling 1.000000 only32 0
cpucycles tracesetup 3 amd64-tscasm precision 118 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-perfevent precision 159 scaling 1.000000 only32 0
cpucycles tracesetup 5 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 6 default-monotonic precision 380 scaling 3.200000 only32 0
cpucycles tracesetup 7 default-gettimeofday precision 3500 scaling 3200.000000 only32 0
cpucycles tracesetup 8 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3200000000
cpucycles implementation amd64-tsc
cpucycles iqm 22 +20+5+11-4+5-4+5-4+5-4+5-4-4+5-4+5-4+5-4+5-4+5-4+5-4-4+5-4+5-4+5-4+5-4+5-4+5-4-4+5-4+5-4+5-4+5-4+5-4+5-4-4+5-4+5-4+5-4+5-4+5-4+5-4
cpucycles observed persecond 1919250000...3900000000 with 8192 loops 3 microseconds
cpucycles observed persecond 2216142857...3136800000 with 16384 loops 6 microseconds
cpucycles observed persecond 2334230769...2768727273 with 32768 loops 12 microseconds
cpucycles observed persecond 2252444444...2437080000 with 65536 loops 26 microseconds
cpucycles observed persecond 2329269230...2424840000 with 131072 loops 51 microseconds
cpucycles observed persecond 2372911764...2421600000 with 262144 loops 101 microseconds
cpucycles observed persecond 2384305418...2408671642 with 524288 loops 202 microseconds
cpucycles observed persecond 2390051851...2402196030 with 1048576 loops 404 microseconds
```

`cfarm26`,
Intel Core i5-4570 in 32-bit mode under KVM,
Debian 12.12,
Linux kernel 6.1.0-41-686-pae:
```
cpucycles version 20251226
cpucycles tracesetup 0 x86-tsc precision 118 scaling 1.000000 only32 0
cpucycles tracesetup 1 x86-tscasm precision 118 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 640 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 2624 scaling 3.192606 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 3492 scaling 3192.606000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3192606000
cpucycles implementation x86-tsc
cpucycles iqm 21 +13+0-3+10-3-3+10-3-3+10-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3+10-3
cpucycles observed persecond 1948250000...6256500000 with 8192 loops 3 microseconds
cpucycles observed persecond 2589500000...5091750000 with 16384 loops 5 microseconds
cpucycles observed persecond 2820454545...3972444445 with 32768 loops 10 microseconds
cpucycles observed persecond 2949380952...3507473685 with 65536 loops 20 microseconds
cpucycles observed persecond 3093325000...3380368422 with 131072 loops 39 microseconds
cpucycles observed persecond 3141984962...3241618321 with 262144 loops 132 microseconds
cpucycles observed persecond 3155211180...3230132076 with 524288 loops 160 microseconds
cpucycles observed persecond 3180147227...3204936661 with 1048576 loops 522 microseconds
```

`cfarm27`,
Intel Core i5-4570 in 32-bit mode under KVM,
Alpine 3.19.9,
Linux kernel 6.6.117-0-lts:
```
cpucycles version 20251226
cpucycles tracesetup 0 x86-tsc precision 118 scaling 1.000000 only32 0
cpucycles tracesetup 1 x86-tscasm precision 118 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 632 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 1156 scaling 3.192606 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 3492 scaling 3192.606000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3192606000
cpucycles implementation x86-tsc
cpucycles iqm 22 +59-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1-4+1+1
cpucycles observed persecond 1834250000...4494500000 with 8192 loops 3 microseconds
cpucycles observed persecond 2444666666...4107000000 with 16384 loops 5 microseconds
cpucycles observed persecond 2932000000...3881750000 with 32768 loops 9 microseconds
cpucycles observed persecond 3073684210...3538647059 with 65536 loops 18 microseconds
cpucycles observed persecond 3113500000...3248790323 with 131072 loops 63 microseconds
cpucycles observed persecond 3156866071...3232554546 with 262144 loops 111 microseconds
cpucycles observed persecond 3178926966...3227636364 with 524288 loops 177 microseconds
cpucycles observed persecond 3181241379...3199993507 with 1048576 loops 463 microseconds
```

`cfarm29`,
IBM POWER9,
Debian 13.1,
Linux kernel 6.12.57+deb13-powerpc64le:
```
cpucycles version 20251226
cpucycles tracesetup 0 ppc64-mftb precision 220 scaling 7.500000 only32 0
cpucycles tracesetup 1 default-perfevent precision 368 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 402 scaling 3.800000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 4100 scaling 3800.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3800000000
cpucycles implementation ppc64-mftb
cpucycles iqm 132 +33-5+3-12+3+3-4-5+3+11-5+3-4-5-4+10-12-4+3-5+11+3-12+3-5+3-4+3-5+3+11-5+3-4-5-4+3-5+3-4+10-4+3-5+3-4+3-5+3-4+3+3+3-5+3-4+3-5+3-4+3+10-4+3
cpucycles observed persecond 2461750000...5299000000 with 2048 loops 3 microseconds
cpucycles observed persecond 2908400000...5135000000 with 4096 loops 4 microseconds
cpucycles observed persecond 3120888888...4116428572 with 8192 loops 8 microseconds
cpucycles observed persecond 3672000000...4298692308 with 16384 loops 14 microseconds
cpucycles observed persecond 3779500000...4077857143 with 32768 loops 29 microseconds
cpucycles observed persecond 3693138888...3923382353 with 65536 loops 35 microseconds
cpucycles observed persecond 3796600000...3914455883 with 131072 loops 69 microseconds
cpucycles observed persecond 3822244604...3881116789 with 262144 loops 138 microseconds
cpucycles observed persecond 3821845323...3851057972 with 524288 loops 277 microseconds
cpucycles observed persecond 3835072202...3849742754 with 1048576 loops 553 microseconds
```

`cfarm45`,
AMD Athlon II X4 640,
Debian 8.11,
Linux kernel 3.16.0-11-686-pae:
```
cpucycles version 20230105
cpucycles tracesetup 0 x86-tsc precision 199 scaling 1.000000 only32 0
cpucycles tracesetup 1 x86-tscasm precision 199 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 170 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 941 scaling 3.000000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 3200 scaling 3000.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3000000000
cpucycles implementation default-perfevent
cpucycles median 72 +12+0+0+0+0+0+0+0+5+0+0+0+0+0+0+0+2+0+0+0+0+0+0+0+1+0+0+0+0+0+0+0+2+0+0+0+0+0+0+0+1+0+0+0+0+0+0+0+2+0+0+0+0+0+0+0+1+0+0+0+0+0+0
cpucycles observed persecond 541500000...1812000000 with 1024 loops 3 microseconds
cpucycles observed persecond 712333333...1212250000 with 2048 loops 5 microseconds
cpucycles observed persecond 1193285714...1733600000 with 4096 loops 6 microseconds
cpucycles observed persecond 1689176470...1804562500 with 8192 loops 33 microseconds
cpucycles observed persecond 1713074626...1770600000 with 16384 loops 66 microseconds
cpucycles observed persecond 1765107692...1795140625 with 32768 loops 129 microseconds
cpucycles observed persecond 1785369649...1800603922 with 65536 loops 256 microseconds
cpucycles observed persecond 1781377862...1796288462 with 131072 loops 261 microseconds
cpucycles observed persecond 1772647398...1778247827 with 262144 loops 691 microseconds
cpucycles observed persecond 1789670493...1794149598 with 524288 loops 870 microseconds
cpucycles observed persecond 1860276211...1861561332 with 1048576 loops 3156 microseconds
```

`cfarm91`,
StarFive JH7100,
Debian trixie/sid,
Linux kernel 5.18.11-starfive:
```
cpucycles version 20250925
cpucycles tracesetup 0 riscv64-rdcycle precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 2702 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 1351 scaling 2.399988 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 2599 scaling 2399.987654 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation default-monotonic
cpucycles iqm 1476 +828-324+60+60+60+60-324+60-324+60+60-324+60+60-324+60+60-324+60-324+60+60-324+62-324+60+60-324+60+60-324+60-324+60+60-324+60+60-324+60-324+60+60-324+60+60-324+60-324+60+60+60+60+60-324+60-324+62+60+444+60-324+60+60
cpucycles observed persecond 1440250000...3968333334 with 1024 loops 7 microseconds
cpucycles observed persecond 1920181818...3072222223 with 2048 loops 10 microseconds
cpucycles observed persecond 2162473684...2733294118 with 4096 loops 18 microseconds
cpucycles observed persecond 2218777777...2530000000 with 8192 loops 35 microseconds
cpucycles observed persecond 2338014705...2490318182 with 16384 loops 67 microseconds
cpucycles observed persecond 2358559701...2440840910 with 32768 loops 133 microseconds
cpucycles observed persecond 2380905660...2419452472 with 65536 loops 264 microseconds
cpucycles observed persecond 2390815939...2410163810 with 131072 loops 526 microseconds
cpucycles observed persecond 2393901140...2403580953 with 262144 loops 1051 microseconds
cpucycles observed persecond 2397546190...2402577217 with 524288 loops 2099 microseconds
cpucycles observed persecond 2398864140...2401570967 with 1048576 loops 4327 microseconds
```

`cfarm92`,
SiFive Freedom U740,
Ubuntu 22.04.3,
Linux kernel 5.19.0-1021-generic:
```
cpucycles version 20240114
cpucycles tracesetup 0 riscv64-rdcycle precision 8 scaling 1.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 2599 scaling 2.399988 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 2599 scaling 2399.987654 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation riscv64-rdcycle
cpucycles median 8 +168+20+2+2+0+0+0+0+570+0+0+0+0+0+0+0+144+0+0+0+0+0+0+0+160+0+0+0+0+0+0+0+160+0+0+0+0+0+0+0+154+0+0+0+0+0+0+0+154+0+0+0+0+0+0+0+152+0+0+0+0+0+0
cpucycles observed persecond 571500000...2198000000 with 1024 loops 3 microseconds
cpucycles observed persecond 833600000...2094000000 with 2048 loops 4 microseconds
cpucycles observed persecond 921888888...1445142858 with 4096 loops 8 microseconds
cpucycles observed persecond 1029625000...1320642858 with 8192 loops 15 microseconds
cpucycles observed persecond 1137034482...1284481482 with 16384 loops 28 microseconds
cpucycles observed persecond 1155701754...1227454546 with 32768 loops 56 microseconds
cpucycles observed persecond 1177464285...1217163637 with 65536 loops 111 microseconds
cpucycles observed persecond 1188018099...1207858448 with 131072 loops 220 microseconds
cpucycles observed persecond 1189925170...1200519363 with 262144 loops 440 microseconds
cpucycles observed persecond 1193962457...1199117446 with 524288 loops 878 microseconds
cpucycles observed persecond 1194051324...1196780111 with 1048576 loops 1811 microseconds
```

`cfarm94`,
SiFive StarFive JH7110,
Alpine 3.22.0_alpha20250108,
Linux kernel 6.12.19-2-lts:
```
cpucycles version 20251226
cpucycles tracesetup 0 riscv64-perfrdcycle precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 riscv64-rdcycle precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 2174 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 1050 scaling 1.500000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 1800 scaling 1500.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1500000000
cpucycles implementation default-monotonic
cpucycles iqm 1125 -375+0+0+0+0+0-375+0+375+0-375+0+0+0+0+0+0-375+0+0+0+0+0-375+0+0+0+0+0+0-375+0+0+0+0+0+0+0-375+0+0+0+0+0-375+0+0+0+0+0-375+0+0+0+0+0+0+0-375+0+0+0+0+0
cpucycles observed persecond 675000000...2500000000 with 1024 loops 4 microseconds
cpucycles observed persecond 875000000...2343750000 with 2048 loops 5 microseconds
cpucycles observed persecond 1041888888...1982428572 with 4096 loops 8 microseconds
cpucycles observed persecond 1232142857...1812500000 with 8192 loops 13 microseconds
cpucycles observed persecond 1350000000...1663043479 with 16384 loops 24 microseconds
cpucycles observed persecond 1451108695...1610818182 with 32768 loops 45 microseconds
cpucycles observed persecond 1459178217...1534111112 with 65536 loops 100 microseconds
cpucycles observed persecond 1478949438...1519193182 with 131072 loops 177 microseconds
cpucycles observed persecond 1492565340...1512880000 with 262144 loops 351 microseconds
cpucycles observed persecond 1493683098...1504258475 with 524288 loops 709 microseconds
cpucycles observed persecond 1498423295...1503488621 with 1048576 loops 1407 microseconds
```

`cfarm95`,
SpacemiT X60,
Debian trixie/sid,
Linux kernel 6.6.36-cfarm #1:
```
cpucycles version 20251226
cpucycles tracesetup 0 riscv64-perfrdcycle precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 riscv64-rdcycle precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 1845 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 365 scaling 1.600000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 1900 scaling 1600.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1600000000
cpucycles implementation default-monotonic
cpucycles iqm 134 +534-2+1+199+66-2+1-1-2+1-1+198+1-1-2+1-1-2+1+199-2+1-1-2+1-1-2+334-2+1-1-2+1-1-2+266+1-1-2+1-1-2+1+531+66+1-1-2+1-1-67+866-69+1-1-2+1-1-2+1-1-2+1-1
cpucycles observed persecond 1400000000...2216750000 with 4096 loops 5 microseconds
cpucycles observed persecond 1405666666...1840200000 with 8192 loops 11 microseconds
cpucycles observed persecond 1568285714...1761473685 with 16384 loops 20 microseconds
cpucycles observed persecond 1563547619...1655000000 with 32768 loops 41 microseconds
cpucycles observed persecond 1564297619...1608975610 with 65536 loops 83 microseconds
cpucycles observed persecond 1589527272...1614343559 with 131072 loops 164 microseconds
cpucycles observed persecond 1594155015...1605333334 with 262144 loops 328 microseconds
cpucycles observed persecond 1595267477...1602367379 with 524288 loops 657 microseconds
cpucycles observed persecond 1597999238...1600842106 with 1048576 loops 1312 microseconds
```

`cfarm103`,
Apple M1 (Icestorm-M1 + Firestorm-M1),
Debian trixie/sid,
Linux kernel 6.5.0-asahi-00780-g62806c2c6f29:
```
cpucycles version 20251226
cpucycles tracesetup 0 arm64-perfpmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 arm64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 arm64-vct precision 186 scaling 86.000000 only32 0
cpucycles tracesetup 3 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-monotonic precision 384 scaling 2.064000 only32 0
cpucycles tracesetup 6 default-gettimeofday precision 2364 scaling 2064.000000 only32 0
cpucycles tracesetup 7 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2064000000
cpucycles implementation arm64-vct
cpucycles iqm 0 +0+0+86+0+0+0+0+0+0+0+86+0+0+0+0+0+0+0+0+86+0+0+0+0+0+0+0+86+0+0+0+0+0+0+0+0+86+0+0+0+0+0+0+0+0+86+0+0+0+0+0+0+0+0+86+0+0+0+0+0+0+0+0+86
cpucycles observed persecond 1290000000...2666000000 with 4096 loops 3 microseconds
cpucycles observed persecond 1651200000...2809333334 with 8192 loops 4 microseconds
cpucycles observed persecond 1825111111...2371142858 with 16384 loops 8 microseconds
cpucycles observed persecond 1927411764...2195866667 with 32768 loops 16 microseconds
cpucycles observed persecond 1991030303...2125032259 with 65536 loops 32 microseconds
cpucycles observed persecond 2017692307...2084476191 with 131072 loops 64 microseconds
cpucycles observed persecond 2047875000...2082428572 with 262144 loops 127 microseconds
cpucycles observed persecond 2056243137...2073177866 with 524288 loops 254 microseconds
cpucycles observed persecond 2060352365...2067378069 with 1048576 loops 612 microseconds
```

`cfarm104`,
Apple M1 (Icestorm-M1 + Firestorm-M1),
MacOSX 12.6 21.6.0:
```
cpucycles version 20251226
cpucycles tracesetup 0 arm64-perfpmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 arm64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 arm64-vct precision 200 scaling 100.000000 only32 0
cpucycles tracesetup 3 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-mach precision 200 scaling 100.000000 only32 0
cpucycles tracesetup 5 default-monotonic precision 2699 scaling 2.399988 only32 0
cpucycles tracesetup 6 default-gettimeofday precision 2699 scaling 2399.987654 only32 0
cpucycles tracesetup 7 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation arm64-vct
cpucycles iqm 0 +4500+0+0+0+0+0+100+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+100+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+100+0+0+0+0+0+0+0
cpucycles observed persecond 1725000000...3450000000 with 8192 loops 3 microseconds
cpucycles observed persecond 1885714285...2640000000 with 16384 loops 6 microseconds
cpucycles observed persecond 2300000000...2533333334 with 32768 loops 22 microseconds
cpucycles observed persecond 2339285714...2523076924 with 65536 loops 27 microseconds
cpucycles observed persecond 2348529411...2421212122 with 131072 loops 67 microseconds
cpucycles observed persecond 2377678571...2420909091 with 262144 loops 111 microseconds
cpucycles observed persecond 2387214611...2410138249 with 524288 loops 218 microseconds
cpucycles observed persecond 2397814207...2411813187 with 1048576 loops 365 microseconds
```

`cfarm110` (`gcc1-power7`),
IBM POWER7,
CentOS 7.9 AltArch,
Linux kernel 3.10.0-1160.105.1.el7.ppc64:
```
cpucycles version 20251226
cpucycles tracesetup 0 ppc64-mftb precision 212 scaling 7.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 226 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 473 scaling 3.550000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 3850 scaling 3550.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3550000000
cpucycles implementation ppc64-mftb
cpucycles iqm 166 -19+44-40+51-47+37-33+37-33+30-26+23-19+16-12+16-12+2+2+2+2-5+9-12+16-19+23-26+30-33+37+9-5-5+9-12+16-12+16-26+30-26+30-33+37-40+44-26+30-33+37-40+44-47+37-33+37-40+30-26+2+2-5+9
cpucycles observed persecond 2427250000...5285000000 with 2048 loops 3 microseconds
cpucycles observed persecond 3185000000...4935000000 with 4096 loops 5 microseconds
cpucycles observed persecond 3381000000...4209333334 with 8192 loops 10 microseconds
cpucycles observed persecond 3360000000...3749052632 with 16384 loops 20 microseconds
cpucycles observed persecond 3533121951...3736205129 with 32768 loops 40 microseconds
cpucycles observed persecond 3499666666...3624393443 with 65536 loops 62 microseconds
cpucycles observed persecond 3541437500...3610281819 with 131072 loops 111 microseconds
cpucycles observed persecond 3555403587...3589828055 with 262144 loops 222 microseconds
cpucycles observed persecond 3570993243...3588260181 with 524288 loops 443 microseconds
cpucycles observed persecond 3578817155...3587444571 with 1048576 loops 885 microseconds
```

`cfarm112` (`gcc2-power8`),
IBM POWER8E,
CentOS 7.9 AltArch,
Linux kernel 3.10.0-1127.13.1.el7.ppc64le:
```
cpucycles version 20251226
cpucycles tracesetup 0 ppc64-mftb precision 266 scaling 7.250000 only32 0
cpucycles tracesetup 1 default-perfevent precision 294 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 665 scaling 3.690000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 3990 scaling 3690.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3690000000
cpucycles implementation ppc64-mftb
cpucycles iqm 213 -10+4-3+12+5+4-17-3+5-18-2+11-2-3-3+5+4-3-2+4+5-3-3-3-10+5-3+5+4-3-10-2-3+4-10+5+4-2-3+4-17+5-3+4+5+4-10+5+4-2+4-3-2-3+4-2-3+12-10-3-3-3+12+5
cpucycles observed persecond 3161000000...10552000000 with 1024 loops 3 microseconds
cpucycles observed persecond 3260428571...4770400000 with 2048 loops 6 microseconds
cpucycles observed persecond 3476416666...4257200000 with 4096 loops 11 microseconds
cpucycles observed persecond 3573041666...3940409091 with 8192 loops 23 microseconds
cpucycles observed persecond 3608704545...3806595239 with 16384 loops 43 microseconds
cpucycles observed persecond 3692062500...3822274194 with 32768 loops 63 microseconds
cpucycles observed persecond 3689968750...3754119048 with 65536 loops 127 microseconds
cpucycles observed persecond 3688804687...3720649607 with 131072 loops 255 microseconds
cpucycles observed persecond 3702731372...3718679134 with 262144 loops 509 microseconds
cpucycles observed persecond 3704490310...3715448544 with 524288 loops 1031 microseconds
cpucycles observed persecond 3709698513...3713976090 with 1048576 loops 1883 microseconds
```

`cfarm120`,
IBM POWER10,
AlmaLinux 9.7,
Linux kernel 5.14.0-427.31.1.el9_4.ppc64le:
```
cpucycles version 20251226
cpucycles tracesetup 0 ppc64-mftb precision 123 scaling 5.750000 only32 0
cpucycles tracesetup 1 default-perfevent precision 206 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 326 scaling 2.950000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 3250 scaling 2950.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2950000000
cpucycles implementation ppc64-mftb
cpucycles iqm 59 -31+16+10-30+16-31+16+16-30+10-31+16+10-30+21-36+16+16-36+16-31+22+10-30+15-36+16+16-36+21-30+10+16-30+10-31+16+10-30+21-30+10+16-30+10-31+16+16-30+15-36+16+16-30+15-30+16+10-30+15-36+16+16-30
cpucycles observed persecond 2286200000...3873333334 with 4096 loops 4 microseconds
cpucycles observed persecond 2597625000...3494166667 with 8192 loops 7 microseconds
cpucycles observed persecond 2857000000...3311538462 with 16384 loops 14 microseconds
cpucycles observed persecond 2796354838...2995931035 with 32768 loops 30 microseconds
cpucycles observed persecond 2874596491...2983927273 with 65536 loops 56 microseconds
cpucycles observed persecond 2933263157...3016337838 with 131072 loops 75 microseconds
cpucycles observed persecond 2941246376...2985897059 with 262144 loops 137 microseconds
cpucycles observed persecond 2930028880...2952029091 with 524288 loops 276 microseconds
cpucycles observed persecond 2935285714...2946295826 with 1048576 loops 552 microseconds
```

`cfarm185`,
Ampere eMAG 8180,
AlmaLinux 8.10,
Linux kernel 4.18.0-553.el8_10.aarch64:
```
cpucycles version 20251226
cpucycles tracesetup 0 arm64-perfpmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 arm64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 arm64-vct precision 175 scaling 75.000000 only32 0
cpucycles tracesetup 3 default-perfevent precision 374 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-monotonic precision 375 scaling 3.000000 only32 0
cpucycles tracesetup 6 default-gettimeofday precision 3300 scaling 3000.000000 only32 0
cpucycles tracesetup 7 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3000000000
cpucycles implementation arm64-vct
cpucycles iqm 42 +33+33+33+33-42-42+33+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33+33-42-42+33+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42+33-42
cpucycles observed persecond 2235000000...3825000000 with 4096 loops 4 microseconds
cpucycles observed persecond 2803125000...3800000000 with 8192 loops 7 microseconds
cpucycles observed persecond 2798437500...3219642858 with 16384 loops 15 microseconds
cpucycles observed persecond 2883870967...3093103449 with 32768 loops 30 microseconds
cpucycles observed persecond 2980000000...3087931035 with 65536 loops 59 microseconds
cpucycles observed persecond 2979375000...3032415255 with 131072 loops 119 microseconds
cpucycles observed persecond 2979062500...3005357143 with 262144 loops 239 microseconds
cpucycles observed persecond 2991527196...3004726891 with 524288 loops 477 microseconds
cpucycles observed persecond 2997720125...3004332984 with 1048576 loops 953 microseconds
```

`cfarm202`,
UltraSparc T5,
Debian forky/sid,
Linux kernel 6.17.0-rc5+:
```
cpucycles version 20251226
cpucycles tracesetup 0 sparc64-rdtick precision 65 scaling 1.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 582 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 515 scaling 3.599910 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 3899 scaling 3599.910000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3599910000
cpucycles implementation sparc64-rdtick
cpucycles iqm 73 +24+24+24+24+0+0+24+0+28+0+0+24+0+0+24+0+0+1+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 2416000000...3769750000 with 4096 loops 5 microseconds
cpucycles observed persecond 3198333333...4216571429 with 8192 loops 8 microseconds
cpucycles observed persecond 3379411764...3871000000 with 16384 loops 16 microseconds
cpucycles observed persecond 3478606060...3722870968 with 32768 loops 32 microseconds
cpucycles observed persecond 3530461538...3651809524 with 65536 loops 64 microseconds
cpucycles observed persecond 3557023255...3617417323 with 131072 loops 128 microseconds
cpucycles observed persecond 3584414062...3614846457 with 262144 loops 255 microseconds
cpucycles observed persecond 3591225048...3606375246 with 524288 loops 510 microseconds
cpucycles observed persecond 3594636630...3602211973 with 1048576 loops 1020 microseconds
```

`cfarm216`,
VM on SPARC-M8,
Solaris 11.4.88.207.1:
```
cpucycles version 20251226
cpucycles tracesetup 0 sparc64-rdtick precision 63 scaling 1.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 504 scaling 5.067000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 5267 scaling 5067.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 5067000000
cpucycles implementation sparc64-rdtick
cpucycles iqm 69 +25+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+41382+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 4115600000...6985666667 with 8192 loops 4 microseconds
cpucycles observed persecond 4562777777...5928571429 with 16384 loops 8 microseconds
cpucycles observed persecond 4558277777...5151812500 with 32768 loops 17 microseconds
cpucycles observed persecond 4821705882...5134937500 with 65536 loops 33 microseconds
cpucycles observed persecond 4966348484...5127546875 with 131072 loops 65 microseconds
cpucycles observed persecond 5003496183...5084015504 with 262144 loops 130 microseconds
cpucycles observed persecond 5003122137...5043069231 with 524288 loops 261 microseconds
cpucycles observed persecond 5002935114...5022831418 with 1048576 loops 523 microseconds
```

`cfarm230`,
Cavium Octeon III V0.2,
Debian 10.13,
Linux kernel 4.9.79-UBNT_E300:
```
cpucycles version 20251226
cpucycles tracesetup 0 mips64-cc precision 323 scaling 1.000000 only32 1
cpucycles tracesetup 1 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 1534 scaling 2.399988 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 2699 scaling 2399.987654 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation mips64-cc
cpucycles iqm 405 +24+26+15+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 862666666...1850000000 with 1024 loops 8 microseconds
cpucycles observed persecond 868764705...1087133334 with 2048 loops 16 microseconds
cpucycles observed persecond 938483870...1051862069 with 4096 loops 30 microseconds
cpucycles observed persecond 979067796...1037964913 with 8192 loops 58 microseconds
cpucycles observed persecond 983837606...1013382609 with 16384 loops 116 microseconds
cpucycles observed persecond 990504310...1005200000 with 32768 loops 231 microseconds
cpucycles observed persecond 996032537...1003843138 with 65536 loops 460 microseconds
cpucycles observed persecond 998830250...1002534352 with 131072 loops 918 microseconds
cpucycles observed persecond 998601196...1000858933 with 262144 loops 1837 microseconds
cpucycles observed persecond 999302205...1000298284 with 524288 loops 3672 microseconds
cpucycles observed persecond 999723381...1000234418 with 1048576 loops 7381 microseconds
```

`cfarm400`,
Loongson-3C5000L-LL,
Debian trixie/sid,
Linux kernel 6.1.0-rc7+:
```
cpucycles version 20251226
cpucycles tracesetup 0 loong64-rdtime precision 124 scaling 24.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 170 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 396 scaling 2.399988 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 2699 scaling 2399.987654 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation loong64-rdtime
cpucycles iqm 48 +24+0+0+24+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+24+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+24+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+24
cpucycles observed persecond 1982400000...3408000000 with 4096 loops 4 microseconds
cpucycles observed persecond 1975200000...2508000000 with 8192 loops 9 microseconds
cpucycles observed persecond 2188000000...2482500000 with 16384 loops 17 microseconds
cpucycles observed persecond 2315294117...2469750000 with 32768 loops 33 microseconds
cpucycles observed persecond 2348776119...2425846154 with 65536 loops 66 microseconds
cpucycles observed persecond 2365714285...2404213741 with 131072 loops 132 microseconds
cpucycles observed persecond 2387467576...2408000000 with 262144 loops 292 microseconds
cpucycles observed persecond 2392334600...2402106871 with 524288 loops 525 microseconds
cpucycles observed persecond 2396800000...2401694657 with 1048576 loops 1049 microseconds
```

`cfarm425`,
Ampere Altra Max M128-30,
Debian forky,
Linux kernel 6.16.12+deb14+1-arm64:
```
cpucycles version 20251226
cpucycles tracesetup 0 arm64-perfpmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 arm64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 arm64-vct precision 196 scaling 96.000000 only32 0
cpucycles tracesetup 3 default-perfevent precision 182 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-monotonic precision 396 scaling 2.399988 only32 0
cpucycles tracesetup 6 default-gettimeofday precision 2699 scaling 2399.987654 only32 0
cpucycles tracesetup 7 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation default-perfevent
cpucycles iqm 72 +37+0+0+0+0+0+0+0+0+13+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+211+0+0+0+0+0+0+0+0+0+0+0+0+0+0+10+0+0+0+0+0+0
cpucycles observed persecond 962800000...1717000000 with 4096 loops 4 microseconds
cpucycles observed persecond 1388000000...2165750000 with 8192 loops 5 microseconds
cpucycles observed persecond 1842777777...2417142858 with 16384 loops 8 microseconds
cpucycles observed persecond 2206066666...2571230770 with 32768 loops 14 microseconds
cpucycles observed persecond 2435814814...2644160000 with 65536 loops 26 microseconds
cpucycles observed persecond 2303982456...2394054546 with 131072 loops 56 microseconds
cpucycles observed persecond 2706288659...2766915790 with 262144 loops 96 microseconds
cpucycles observed persecond 2747863874...2778698413 with 524288 loops 190 microseconds
cpucycles observed persecond 2776529100...2792220745 with 1048576 loops 377 microseconds
```

`cfarm430`,
AMD EPYC 7773X,
FreeBSD 16.0 VM:
```
cpucycles version 20251226
cpucycles tracesetup 0 amd64-perfpmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 amd64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 amd64-tsc precision 122 scaling 1.000000 only32 0
cpucycles tracesetup 3 amd64-tscasm precision 122 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 6 default-monotonic precision 468 scaling 2.399988 only32 0
cpucycles tracesetup 7 default-gettimeofday precision 2699 scaling 2399.987654 only32 0
cpucycles tracesetup 8 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation amd64-tsc
cpucycles iqm 30 +14+14-8-8+14-8+14-8-8+630+14-8-8+14-8+14-8-8+14-8+14-8+14-8-8+14-8+14-8+14-8-8+14-8+14-8+14-8-8+14-8+14-8+14-8-8+14-8+14-8+14-8-8+14-8+14-8-8+14-8+14-8-8+14
cpucycles observed persecond 1557600000...2720666667 with 8192 loops 4 microseconds
cpucycles observed persecond 1949750000...2665666667 with 16384 loops 7 microseconds
cpucycles observed persecond 2078266666...2426769231 with 32768 loops 14 microseconds
cpucycles observed persecond 2078266666...2240071429 with 65536 loops 29 microseconds
cpucycles observed persecond 2132058823...2222666667 with 131072 loops 67 microseconds
cpucycles observed persecond 2168434782...2210318585 with 262144 loops 114 microseconds
cpucycles observed persecond 2196995594...2218284445 with 524288 loops 226 microseconds
cpucycles observed persecond 2196947136...2207446903 with 1048576 loops 453 microseconds
```

IBM z15:
```
cpucycles version 20230106
cpucycles tracesetup 0 s390x-stckf precision 250 scaling 1.269531 only32 0
cpucycles tracesetup 1 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 272 scaling 5.200000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 5400 scaling 5200.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 5200000000
cpucycles implementation s390x-stckf
cpucycles median 48 +87+8+0-2+0+0+38-2+0+1-3+1+28+0+3-3+1+0+28+0-2+3+0-2+36+0+0+0+1+0+28+0-2+0+3-2+35+1+0-2+0+3+28+0-2+0+0-2+3+25+3+0-2+0+1+35+1+0+0-2+0+28+0
cpucycles observed persecond 4948941176...5627733334 with 8192 loops 16 microseconds
cpucycles observed persecond 4104125000...5515666667 with 16384 loops 7 microseconds
cpucycles observed persecond 5047076923...5987818182 with 32768 loops 12 microseconds
cpucycles observed persecond 5044846153...5475708334 with 65536 loops 25 microseconds
cpucycles observed persecond 5141313725...5357428572 with 131072 loops 50 microseconds
cpucycles observed persecond 5150892156...5257250000 with 262144 loops 101 microseconds
cpucycles observed persecond 5183421568...5236549505 with 524288 loops 203 microseconds
cpucycles observed persecond 5190282555...5216582717 with 1048576 loops 406 microseconds
```
